package com.feingto.iot.server.service;

import com.feingto.iot.common.model.mqtt.SendMessage;
import com.feingto.iot.common.model.mqtt.SubscribeMessage;
import com.feingto.iot.common.service.mqtt.MessageRequest;
import com.feingto.iot.server.cache.SessionCache;
import com.feingto.iot.server.cache.SubscribeCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.IgniteMessaging;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Mqtt 消息推送服务
 *
 * @author longfei
 */
@Slf4j
@Service
public class PushService {
    private static final String LISTEN_TOPIC = "internal-mqtt-topic";

    @Resource(name = "igniteSubscribe")
    private IgniteCache<String, ConcurrentHashMap<String, SubscribeMessage>> igniteSubscribe;

    @Resource(name = "igniteMessaging")
    private IgniteMessaging igniteMessaging;

    @PostConstruct
    public void initListen() {
        igniteMessaging.localListen(LISTEN_TOPIC, (nodeId, msg) -> {
            log.debug(">>> listen from cluster group nodeId: {}", nodeId);
            this.push((SendMessage) msg);
            return true;
        });
    }

    /**
     * 发布集群消息
     */
    public void internalSend(SendMessage mqttMessage) {
        if (CollectionUtils.isNotEmpty(igniteMessaging.clusterGroup().nodes())) {
            log.debug(">>> send to cluster group");
            igniteMessaging.send(LISTEN_TOPIC, mqttMessage);
        } else {
            this.push(mqttMessage);
        }
    }

    /**
     * 推送消息
     */
    private void push(SendMessage message) {
        SubscribeCache.getInstance(igniteSubscribe).findByTopic(message.topic())
                .forEach(subscribe -> Optional.ofNullable(SessionCache.getInstance()
                        .get(subscribe.clientId()))
                        .ifPresent(sess -> {
                            log.debug(">>> send to {} with topic {}", subscribe.clientId(), message.topic());
                            MessageRequest.publish(sess.channel(), message);
                        }));
    }
}
