/*
 * 02/11/2008
 *
 * JSPTokenMaker.java - Generates tokens for JSP syntax highlighting.
 * 
 * This library is distributed under a modified BSD license.  See the included
 * LICENSE file for details.
 */
package org.fife.ui.rsyntaxtextarea.modes;

import java.io.*;
import javax.swing.text.Segment;
import java.util.Stack;

import org.fife.ui.rsyntaxtextarea.*;


/**
 * Scanner for JSP files (supporting HTML 5).
 *
 * This implementation was created using
 * <a href="https://www.jflex.de/">JFlex</a> 1.4.1; however, the generated file
 * was modified for performance.  Memory allocation needs to be almost
 * completely removed to be competitive with the handwritten lexers (subclasses
 * of <code>AbstractTokenMaker</code>), so this class has been modified so that
 * Strings are never allocated (via yytext()), and the scanner never has to
 * worry about refilling its buffer (needlessly copying chars around).
 * We can achieve this because RText always scans exactly 1 line of tokens at a
 * time, and hands the scanner this line as an array of characters (a Segment
 * really).  Since tokens contain pointers to char arrays instead of Strings
 * holding their contents, there is no need for allocating new memory for
 * Strings.<p>
 *
 * The actual algorithm generated for scanning has, of course, not been
 * modified.<p>
 *
 * If you wish to regenerate this file yourself, keep in mind the following:
 * <ul>
 *   <li>The generated <code>JSPTokenMaker.java</code> file will contain two
 *       definitions of both <code>zzRefill</code> and <code>yyreset</code>.
 *       You should hand-delete the second of each definition (the ones
 *       generated by the lexer), as these generated methods modify the input
 *       buffer, which we'll never have to do.</li>
 *   <li>You should also change the declaration/definition of zzBuffer to NOT
 *       be initialized.  This is a needless memory allocation for us since we
 *       will be pointing the array somewhere else anyway.</li>
 *   <li>You should NOT call <code>yylex()</code> on the generated scanner
 *       directly; rather, you should use <code>getTokenList</code> as you would
 *       with any other <code>TokenMaker</code> instance.</li>
 * </ul>
 *
 * @author Robert Futrell
 * @version 0.9
 */
%%

%public
%class JSPTokenMaker
%extends AbstractMarkupTokenMaker
%unicode
%type org.fife.ui.rsyntaxtextarea.Token


%{

	/**
	 * Type specific to JSPTokenMaker denoting a line ending with an unclosed
	 * double-quote attribute.
	 */
	static final int INTERNAL_ATTR_DOUBLE			= -1;


	/**
	 * Type specific to JSPTokenMaker denoting a line ending with an unclosed
	 * single-quote attribute.
	 */
	static final int INTERNAL_ATTR_SINGLE			= -2;


	/**
	 * Token type specific to JSPTokenMaker; this signals that the user has
	 * ended a line with an unclosed HTML tag; thus a new line is beginning
	 * still inside of the tag.
	 */
	static final int INTERNAL_INTAG					= -3;

	/**
	 * Token type specific to JSPTokenMaker; this signals that the user has
	 * ended a line with an unclosed <code>&lt;script&gt;</code> tag.
	 */
	static final int INTERNAL_INTAG_SCRIPT			= -4;

	/**
	 * Token type specifying we're in a double-quoted attribute in a
	 * script tag.
	 */
	static final int INTERNAL_ATTR_DOUBLE_QUOTE_SCRIPT = -5;

	/**
	 * Token type specifying we're in a single-quoted attribute in a
	 * script tag.
	 */
	static final int INTERNAL_ATTR_SINGLE_QUOTE_SCRIPT = -6;

	/**
	 * Token type specifying that the user has
	 * ended a line with an unclosed <code>&lt;style&gt;</code> tag.
	 */
	static final int INTERNAL_INTAG_STYLE			= -7;

	/**
	 * Token type specifying we're in a double-quoted attribute in a
	 * style tag.
	 */
	static final int INTERNAL_ATTR_DOUBLE_QUOTE_STYLE = -8;

	/**
	 * Token type specifying we're in a single-quoted attribute in a
	 * style tag.
	 */
	static final int INTERNAL_ATTR_SINGLE_QUOTE_STYLE = -9;

	/**
	 * Token type specifying we're in a JSP hidden comment ("<%-- ... --%>").
	 */
	private static final int INTERNAL_IN_HIDDEN_COMMENT		= -10;

	/**
	 * Token type specifying we're in a JSP directive (either include, page
	 * or taglib).
	 */
	private static final int INTERNAL_IN_JSP_DIRECTIVE			= -11;

	/**
	 * Token type specifying we're in JavaScript.
	 */
	static final int INTERNAL_IN_JS					= -12;

	/**
	 * Token type specifying we're in a JavaScript multi-line comment.
	 */
	static final int INTERNAL_IN_JS_MLC				= -13;

	/**
	 * Token type specifying we're in an invalid multi-line JS string.
	 */
	static final int INTERNAL_IN_JS_STRING_INVALID	= -14;

	/**
	 * Token type specifying we're in a valid multi-line JS string.
	 */
	static final int INTERNAL_IN_JS_STRING_VALID		= -15;

	/**
	 * Token type specifying we're in an invalid multi-line JS single-quoted string.
	 */
	static final int INTERNAL_IN_JS_CHAR_INVALID	= -16;

	/**
	 * Token type specifying we're in a valid multi-line JS single-quoted string.
	 */
	static final int INTERNAL_IN_JS_CHAR_VALID		= -17;

	/**
	 * Internal type denoting a line ending in CSS.
	 */
	static final int INTERNAL_CSS					= -18;

	/**
	 * Internal type denoting a line ending in a CSS property.
	 */
	static final int INTERNAL_CSS_PROPERTY			= -19;

	/**
	 * Internal type denoting a line ending in a CSS property value.
	 */
	static final int INTERNAL_CSS_VALUE				= -20;

	/**
	 * Token type specifying we're in a valid multi-line template literal.
	 */
	static final int INTERNAL_IN_JS_TEMPLATE_LITERAL_VALID = -23;

	/**
	 * Token type specifying we're in an invalid multi-line template literal.
	 */
	static final int INTERNAL_IN_JS_TEMPLATE_LITERAL_INVALID = -24;

	/**
	 * Internal type denoting line ending in a CSS double-quote string.
	 * The state to return to is embedded in the actual end token type.
	 */
	static final int INTERNAL_CSS_STRING				= -(1<<11);

	/**
	 * Internal type denoting line ending in a CSS single-quote string.
	 * The state to return to is embedded in the actual end token type.
	 */
	static final int INTERNAL_CSS_CHAR				= -(2<<11);

	/**
	 * Internal type denoting line ending in a CSS multi-line comment.
	 * The state to return to is embedded in the actual end token type.
	 */
	static final int INTERNAL_CSS_MLC				= -(3<<11);

	/**
	 * Token type specifying we're in a Java documentation comment.
	 */
	static final int INTERNAL_IN_JAVA_DOCCOMMENT		= -(4<<11);

	/**
	 * Token type specifying we're in Java code.
	 */
	static final int INTERNAL_IN_JAVA_EXPRESSION		= -(5<<11);

	/**
	 * Token type specifying we're in Java multiline comment.
	 */
	static final int INTERNAL_IN_JAVA_MLC			= -(6<<11);

	/**
	 * The state previous CSS-related state we were in before going into a CSS
	 * string, multi-line comment, etc.
	 */
	private int cssPrevState;

	/**
	 * The state JSP was started in (YYINITIAL, INTERNAL_IN_JS, etc.).
	 */
	private int jspInState;

	/**
	 * Whether closing markup tags are automatically completed for JSP.
	 */
	private static boolean completeCloseTags;

	/**
	 * When in the JS_STRING state, whether the current string is valid.
	 */
	private boolean validJSString;

	/**
	 * Language state set on HTML tokens.  Must be 0.
	 */
	static final int LANG_INDEX_DEFAULT = 0;

	/**
	 * Language state set on JavaScript tokens.
	 */
	static final int LANG_INDEX_JS = 1;

	/**
	 * Language state set on CSS tokens.
	 */
	static final int LANG_INDEX_CSS = 2;

	private Stack<Boolean> varDepths;


	/**
	 * Constructor.  This must be here because JFlex does not generate a
	 * no-parameter constructor.
	 */
	public JSPTokenMaker() {
		super();
	}


	/**
	 * Adds the token specified to the current linked list of tokens as an
	 * "end token;" that is, at <code>zzMarkedPos</code>.
	 *
	 * @param tokenType The token's type.
	 */
	private void addEndToken(int tokenType) {
		addToken(zzMarkedPos,zzMarkedPos, tokenType);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param tokenType The token's type.
	 * @see #addToken(int, int, int)
	 */
	private void addHyperlinkToken(int start, int end, int tokenType) {
		int so = start + offsetShift;
		addToken(zzBuffer, start,end, tokenType, so, true);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param tokenType The token's type.
	 */
	private void addToken(int tokenType) {
		addToken(zzStartRead, zzMarkedPos-1, tokenType);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param tokenType The token's type.
	 */
	private void addToken(int start, int end, int tokenType) {
		int so = start + offsetShift;
		addToken(zzBuffer, start,end, tokenType, so);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param array The character array.
	 * @param start The starting offset in the array.
	 * @param end The ending offset in the array.
	 * @param tokenType The token's type.
	 * @param startOffset The offset in the document at which this token
	 *                    occurs.
	 */
	@Override
	public void addToken(char[] array, int start, int end, int tokenType, int startOffset) {
		super.addToken(array, start,end, tokenType, startOffset);
		zzStartRead = zzMarkedPos;
	}


	@Override
	protected OccurrenceMarker createOccurrenceMarker() {
		return new HtmlOccurrenceMarker();
	}


	/**
	 * Sets whether markup close tags should be completed.  You might not want
	 * this to be the case, since some tags in standard HTML aren't usually
	 * closed.
	 *
	 * @return Whether closing markup tags are completed.
	 * @see #setCompleteCloseTags(boolean)
	 */
	@Override
	public boolean getCompleteCloseTags() {
		return completeCloseTags;
	}


	@Override
	public boolean getCurlyBracesDenoteCodeBlocks(int languageIndex) {
		return languageIndex==LANG_INDEX_CSS || languageIndex==LANG_INDEX_JS;
	}


	@Override
	public String[] getLineCommentStartAndEnd(int languageIndex) {
		switch (languageIndex) {
			case LANG_INDEX_JS:
				return new String[] { "//", null };
			case LANG_INDEX_CSS:
				return new String[] { "/*", "*/" };
			default:
				return new String[] { "<!--", "-->" };
		}
	}


	@Override
	public boolean getMarkOccurrencesOfTokenType(int type) {
		return type==Token.FUNCTION || type==Token.VARIABLE ||
				type==Token.MARKUP_TAG_NAME;
	}


	/**
	 * Overridden to handle newlines in JS and CSS differently than those in
	 * markup.
	 */
	@Override
	public boolean getShouldIndentNextLineAfter(Token token) {
		int languageIndex = token==null ? 0 : token.getLanguageIndex();
		if (getCurlyBracesDenoteCodeBlocks(languageIndex)) {
			if (token!=null && token.length()==1) {
				char ch = token.charAt(0);
				return ch=='{' || ch=='(';
			}
		}
		return false;
	}


	/**
	 * Returns the first token in the linked list of tokens generated
	 * from <code>text</code>.  This method must be implemented by
	 * subclasses so they can correctly implement syntax highlighting.
	 *
	 * @param text The text from which to get tokens.
	 * @param initialTokenType The token type we should start with.
	 * @param startOffset The offset into the document at which
	 *        <code>text</code> starts.
	 * @return The first <code>Token</code> in a linked list representing
	 *         the syntax highlighted text.
	 */
	@Override
	public Token getTokenList(Segment text, int initialTokenType, int startOffset) {

		resetTokenList();
		this.offsetShift = -text.offset + startOffset;
		jspInState = YYINITIAL; // Shouldn't be necessary
		cssPrevState = CSS; // Shouldn't be necessary
		int languageIndex = 0;

		// Start off in the proper state.
		int state = Token.NULL;
		switch (initialTokenType) {
			case Token.MARKUP_COMMENT:
				state = COMMENT;
				break;
			case Token.PREPROCESSOR:
				state = PI;
				break;
			case Token.VARIABLE:
				state = DTD;
				break;
			case INTERNAL_INTAG:
				state = INTAG;
				break;
			case INTERNAL_INTAG_SCRIPT:
				state = INTAG_SCRIPT;
				break;
			case INTERNAL_INTAG_STYLE:
				state = INTAG_STYLE;
				break;
			case INTERNAL_ATTR_DOUBLE:
				state = INATTR_DOUBLE;
				break;
			case INTERNAL_ATTR_SINGLE:
				state = INATTR_SINGLE;
				break;
			case INTERNAL_ATTR_DOUBLE_QUOTE_SCRIPT:
				state = INATTR_DOUBLE_SCRIPT;
				break;
			case INTERNAL_ATTR_SINGLE_QUOTE_SCRIPT:
				state = INATTR_SINGLE_SCRIPT;
				break;
			case INTERNAL_ATTR_DOUBLE_QUOTE_STYLE:
				state = INATTR_DOUBLE_STYLE;
				break;
			case INTERNAL_ATTR_SINGLE_QUOTE_STYLE:
				state = INATTR_SINGLE_STYLE;
				break;
			case INTERNAL_IN_HIDDEN_COMMENT:
				state = HIDDEN_COMMENT;
				break;
			case INTERNAL_IN_JSP_DIRECTIVE:
				state = JSP_DIRECTIVE;
				break;
			case INTERNAL_IN_JS:
				state = JAVASCRIPT;
				languageIndex = LANG_INDEX_JS;
				break;
			case INTERNAL_IN_JS_MLC:
				state = JS_MLC;
				languageIndex = LANG_INDEX_JS;
				break;
			case INTERNAL_IN_JS_STRING_INVALID:
				state = JS_STRING;
				languageIndex = LANG_INDEX_JS;
				validJSString = false;
				break;
			case INTERNAL_IN_JS_STRING_VALID:
				state = JS_STRING;
				languageIndex = LANG_INDEX_JS;
				validJSString = true;
				break;
			case INTERNAL_IN_JS_CHAR_INVALID:
				state = JS_CHAR;
				languageIndex = LANG_INDEX_JS;
				validJSString = false;
				break;
			case INTERNAL_IN_JS_CHAR_VALID:
				state = JS_CHAR;
				languageIndex = LANG_INDEX_JS;
				validJSString = true;
				break;
			case INTERNAL_CSS:
				state = CSS;
				languageIndex = LANG_INDEX_CSS;
				break;
			case INTERNAL_CSS_PROPERTY:
				state = CSS_PROPERTY;
				languageIndex = LANG_INDEX_CSS;
				break;
			case INTERNAL_CSS_VALUE:
				state = CSS_VALUE;
				languageIndex = LANG_INDEX_CSS;
				break;
			case INTERNAL_IN_JS_TEMPLATE_LITERAL_VALID:
				state = JS_TEMPLATE_LITERAL;
				languageIndex = LANG_INDEX_JS;
				validJSString = true;
				break;
			case INTERNAL_IN_JS_TEMPLATE_LITERAL_INVALID:
				state = JS_TEMPLATE_LITERAL;
				languageIndex = LANG_INDEX_JS;
				validJSString = false;
				break;
			default:
				if (initialTokenType<-1024) {
					// INTERNAL_IN_JAVAxxx - jspInState or
					// INTERNAL_IN_CSSxxx - cssPrevState
					int main = -(-initialTokenType & 0xffffff00);
					switch (main) {
						default: // Should never happen
						case INTERNAL_IN_JAVA_DOCCOMMENT:
							state = JAVA_DOCCOMMENT;
							jspInState = -initialTokenType&0xff;
							break;
						case INTERNAL_IN_JAVA_EXPRESSION:
							state = JAVA_EXPRESSION;
							jspInState = -initialTokenType&0xff;
							break;
						case INTERNAL_IN_JAVA_MLC:
							state = JAVA_MLC;
							jspInState = -initialTokenType&0xff;
							break;
						case INTERNAL_CSS_STRING:
							state = CSS_STRING;
							languageIndex = LANG_INDEX_CSS;
							cssPrevState = -initialTokenType&0xff;
							break;
						case INTERNAL_CSS_CHAR:
							state = CSS_CHAR_LITERAL;
							languageIndex = LANG_INDEX_CSS;
							cssPrevState = -initialTokenType&0xff;
							break;
						case INTERNAL_CSS_MLC:
							state = CSS_C_STYLE_COMMENT;
							languageIndex = LANG_INDEX_CSS;
							cssPrevState = -initialTokenType&0xff;
							break;
					}
				}
				else {
					state = Token.NULL;
				}
				break;
		}

		setLanguageIndex(languageIndex);
		start = text.offset;
		s = text;
		try {
			yyreset(zzReader);
			yybegin(state);
			return yylex();
		} catch (IOException ioe) {
			ioe.printStackTrace();
			return new TokenImpl();
		}

	}


	/**
	 * Overridden to accept letters, digits, underscores, and hyphens.
	 */
	@Override
	public boolean isIdentifierChar(int languageIndex, char ch) {
		return Character.isLetterOrDigit(ch) || ch=='-' || ch=='.' || ch=='_';
	}


	/**
	 * Sets whether markup close tags should be completed.  You might not want
	 * this to be the case, since some tags in standard HTML aren't usually
	 * closed.
	 *
	 * @param complete Whether closing markup tags are completed.
	 * @see #getCompleteCloseTags()
	 */
	public static void setCompleteCloseTags(boolean complete) {
		completeCloseTags = complete;
	}


	/**
	 * Refills the input buffer.
	 *
	 * @return      <code>true</code> if EOF was reached, otherwise
	 *              <code>false</code>.
	 */
	private boolean zzRefill() {
		return zzCurrentPos>=s.offset+s.count;
	}


	/**
	 * Resets the scanner to read from a new input stream.
	 * Does not close the old reader.
	 *
	 * All internal variables are reset, the old input stream 
	 * <b>cannot</b> be reused (internal buffer is discarded and lost).
	 * Lexical state is set to <tt>YY_INITIAL</tt>.
	 *
	 * @param reader   the new input stream 
	 */
	public final void yyreset(Reader reader) {
		// 's' has been updated.
		zzBuffer = s.array;
		/*
		 * We replaced the line below with the two below it because zzRefill
		 * no longer "refills" the buffer (since the way we do it, it's always
		 * "full" the first time through, since it points to the segment's
		 * array).  So, we assign zzEndRead here.
		 */
		//zzStartRead = zzEndRead = s.offset;
		zzStartRead = s.offset;
		zzEndRead = zzStartRead + s.count - 1;
		zzCurrentPos = zzMarkedPos = zzPushbackPos = s.offset;
		zzLexicalState = YYINITIAL;
		zzReader = reader;
		zzAtBOL  = true;
		zzAtEOF  = false;
	}


%}

// HTML-specific stuff.
Whitespace					= ([ \t\f])
LineTerminator				= ([\n])
Identifier					= ([^ \t\n<&]+)
EntityReference				= ([&][^; \t]*[;]?)
InTagIdentifier				= ([^ \t\n\"\'/=>]+)
UnclosedStringLiteral		= ([\"][^\"]*)
StringLiteral				= ({UnclosedStringLiteral}[\"])
UnclosedCharLiteral			= ([\'][^\']*)
CharLiteral					= ({UnclosedCharLiteral}[\'])
EndScriptTag				= ("</" [sS][cC][rR][iI][pP][tT] ">")
EndStyleTag					= ("</" [sS][tT][yY][lL][eE] ">")

JspExpressionStart		= ("<%=")
JspScriptletStart		= ("<%")
JspDeclarationStart		= ("<%!")
JspStart				= ({JspExpressionStart}|{JspScriptletStart}|{JspDeclarationStart})

// General stuff.
Letter							= [A-Za-z]
NonzeroDigit					= [1-9]
BinaryDigit						= ([0-1])
Digit							= ("0"|{NonzeroDigit})
HexDigit						= ({Digit}|[A-Fa-f])
OctalDigit						= ([0-7])
LetterOrUnderscore				= ({Letter}|"_")
LetterOrUnderscoreOrDash		= ({LetterOrUnderscore}|[\-])


// Java stuff.
AnyCharacterButApostropheOrBackSlash	= ([^\\'])
AnyCharacterButDoubleQuoteOrBackSlash	= ([^\\\"\n])
EscapedSourceCharacter				= ("u"{HexDigit}{HexDigit}{HexDigit}{HexDigit})
Escape							= ("\\"(([btnfr\"'\\])|([0123]{OctalDigit}?{OctalDigit}?)|({OctalDigit}{OctalDigit}?)|{EscapedSourceCharacter}))
NonSeparator						= ([^\t\f\r\n\ \(\)\{\}\[\]\;\,\.\=\>\<\!\~\?\:\+\-\*\/\&\|\^\%\"\'\`]|"#"|"\\")
IdentifierStart					= ({LetterOrUnderscore}|"$")
IdentifierPart						= ({IdentifierStart}|{Digit}|("\\"{EscapedSourceCharacter}))
WhiteSpace				= ([ \t\f])
JCharLiteral				= ([\']({AnyCharacterButApostropheOrBackSlash}|{Escape})[\'])
JUnclosedCharLiteral			= ([\'][^\'\n]*)
JErrorCharLiteral			= ({UnclosedCharLiteral}[\'])
JStringLiteral				= ([\"]({AnyCharacterButDoubleQuoteOrBackSlash}|{Escape})*[\"])
JUnclosedStringLiteral		= ([\"]([\\].|[^\\\"])*[^\"]?)
JErrorStringLiteral			= ({UnclosedStringLiteral}[\"])

MLCBegin					= "/*"
MLCEnd					= "*/"
DocCommentBegin			= "/**"
LineCommentBegin			= "//"

DigitOrUnderscore			= ({Digit}|[_])
DigitsAndUnderscoresEnd		= ({DigitOrUnderscore}*{Digit})
IntegerHelper				= (({NonzeroDigit}{DigitsAndUnderscoresEnd}?)|"0")
IntegerLiteral				= ({IntegerHelper}[lL]?)

BinaryDigitOrUnderscore		= ({BinaryDigit}|[_])
BinaryDigitsAndUnderscores	= ({BinaryDigit}({BinaryDigitOrUnderscore}*{BinaryDigit})?)
BinaryLiteral				= ("0"[bB]{BinaryDigitsAndUnderscores})

HexDigitOrUnderscore		= ({HexDigit}|[_])
HexDigitsAndUnderscores		= ({HexDigit}({HexDigitOrUnderscore}*{HexDigit})?)
OctalDigitOrUnderscore		= ({OctalDigit}|[_])
OctalDigitsAndUnderscoresEnd= ({OctalDigitOrUnderscore}*{OctalDigit})
HexHelper					= ("0"(([xX]{HexDigitsAndUnderscores})|({OctalDigitsAndUnderscoresEnd})))
HexLiteral					= ({HexHelper}[lL]?)

FloatHelper1				= ([fFdD]?)
FloatHelper2				= ([eE][+-]?{Digit}+{FloatHelper1})
FloatLiteral1				= ({Digit}+"."({FloatHelper1}|{FloatHelper2}|{Digit}+({FloatHelper1}|{FloatHelper2})))
FloatLiteral2				= ("."{Digit}+({FloatHelper1}|{FloatHelper2}))
FloatLiteral3				= ({Digit}+{FloatHelper2})
FloatLiteral				= ({FloatLiteral1}|{FloatLiteral2}|{FloatLiteral3}|({Digit}+[fFdD]))

ErrorNumberFormat			= (({IntegerLiteral}|{HexLiteral}|{FloatLiteral}){NonSeparator}+)
BooleanLiteral				= ("true"|"false")
Separator					= ([\(\)\{\}\[\]])
Separator2				= ([\;,.])
NonAssignmentOperator		= ("+"|"-"|"<="|"^"|"++"|"<"|"*"|">="|"%"|"--"|">"|"/"|"!="|"?"|">>"|"!"|"&"|"=="|":"|">>"|"~"|"|"|"||"|"&&"|">>>")
AssignmentOperator			= ("="|"-="|"*="|"/="|"|="|"&="|"^="|"+="|"%="|"<<="|">>="|">>>=")
Operator					= ({NonAssignmentOperator}|{AssignmentOperator})
JIdentifier				= ({IdentifierStart}{IdentifierPart}*)
ErrorIdentifier			= (({NonSeparator}|\`)+)
Annotation				= ("@"{JIdentifier}?)
PrimitiveTypes				= ("boolean"|"byte"|"char"|"double" |"float"|"int"|"long"|"short")

CurrentBlockTag				= ("author"|"deprecated"|"exception"|"param"|"return"|"see"|"serial"|"serialData"|"serialField"|"since"|"throws"|"version")
ProposedBlockTag			= ("category"|"example"|"tutorial"|"index"|"exclude"|"todo"|"internal"|"obsolete"|"threadsafety")
BlockTag					= ({CurrentBlockTag}|{ProposedBlockTag})
InlineTag					= ("code"|"docRoot"|"inheritDoc"|"link"|"linkplain"|"literal"|"value")

URLGenDelim				= ([:\/\?#\[\]@])
URLSubDelim				= ([\!\$&'\(\)\*\+,;=])
URLUnreserved			= ({LetterOrUnderscore}|{Digit}|[\-\.\~])
URLCharacter			= ({URLGenDelim}|{URLSubDelim}|{URLUnreserved}|[%])
URLCharacters			= ({URLCharacter}*)
URLEndCharacter			= ([\/\$]|{Letter}|{Digit})
URL						= (((https?|f(tp|ile))"://"|"www.")({URLCharacters}{URLEndCharacter})?)

// JavaScript stuff.
JS_MLCBegin				= ({MLCBegin})
JS_MLCEnd					= ({MLCEnd})
JS_LineCommentBegin			= ({LineCommentBegin})
JS_IntegerLiteral			= ({IntegerLiteral})
JS_HexLiteral				= ({HexLiteral})
JS_FloatLiteral			= ({FloatLiteral})
JS_ErrorNumberFormat		= ({ErrorNumberFormat})
JS_Separator				= ({Separator})
JS_Separator2				= ({Separator2})
JS_Operator				= ({Operator})
JS_Identifier				= ({JIdentifier})
JS_ErrorIdentifier			= ({NonSeparator}+)
JS_Regex					= ("/"([^\*\\/]|\\.)([^/\\]|\\.)*"/"[gim]*)

JS_TemplateLiteralExprStart	= ("${")


// CSS stuff.
CSS_SelectorPiece			= (("*"|"."|{LetterOrUnderscoreOrDash})({LetterOrUnderscoreOrDash}|"."|{Digit})*)
CSS_PseudoClass				= (":"("root"|"nth-child"|"nth-last-child"|"nth-of-type"|"nth-last-of-type"|"first-child"|"last-child"|"first-of-type"|"last-of-type"|"only-child"|"only-of-type"|"empty"|"link"|"visited"|"active"|"hover"|"focus"|"target"|"lang"|"enabled"|"disabled"|"checked"|":first-line"|":first-letter"|":before"|":after"|"not"))
CSS_AtKeyword				= ("@"{CSS_SelectorPiece})
CSS_Id						= ("#"{CSS_SelectorPiece})
CSS_Separator				= ([;\(\)\[\]])
CSS_MlcStart				= ({JS_MLCBegin})
CSS_MlcEnd					= ({JS_MLCEnd})
CSS_Property				= ([\*]?{LetterOrUnderscoreOrDash}({LetterOrUnderscoreOrDash}|{Digit})*)
CSS_ValueChar				= ({LetterOrUnderscoreOrDash}|[\\/])
CSS_Value					= ({CSS_ValueChar}*)
CSS_Function				= ({CSS_Value}\()
CSS_Digits					= ([\-]?{Digit}+([0-9\.]+)?(pt|pc|in|mm|cm|em|ex|px|ms|s|%)?)
CSS_Hex						= ("#"[0-9a-fA-F]+)
CSS_Number					= ({CSS_Digits}|{CSS_Hex})


%state COMMENT
%state PI
%state DTD
%state INTAG
%state INTAG_CHECK_TAG_NAME
%state INATTR_DOUBLE
%state INATTR_SINGLE
%state INTAG_SCRIPT
%state INATTR_DOUBLE_SCRIPT
%state INATTR_SINGLE_SCRIPT
%state INTAG_STYLE
%state INATTR_DOUBLE_STYLE
%state INATTR_SINGLE_STYLE
%state JAVASCRIPT
%state JS_STRING
%state JS_CHAR
%state JS_MLC
%state JS_EOL_COMMENT
%state HIDDEN_COMMENT
%state JAVA_DOCCOMMENT
%state JAVA_EXPRESSION
%state JAVA_MLC
%state JSP_DIRECTIVE
%state CSS
%state CSS_PROPERTY
%state CSS_VALUE
%state CSS_STRING
%state CSS_CHAR_LITERAL
%state CSS_C_STYLE_COMMENT
%state JS_TEMPLATE_LITERAL
%state JS_TEMPLATE_LITERAL_EXPR


%%

<YYINITIAL> {
	"<!--"						{ start = zzMarkedPos-4; yybegin(COMMENT); }
	"<"[sS][cC][rR][iI][pP][tT]	{
								  addToken(zzStartRead,zzStartRead, Token.MARKUP_TAG_DELIMITER);
								  addToken(zzMarkedPos-6,zzMarkedPos-1, Token.MARKUP_TAG_NAME);
								  start = zzMarkedPos; yybegin(INTAG_SCRIPT);
								}
	"<"[sS][tT][yY][lL][eE]		{
								  addToken(zzStartRead,zzStartRead, Token.MARKUP_TAG_DELIMITER);
								  addToken(zzMarkedPos-5,zzMarkedPos-1, Token.MARKUP_TAG_NAME);
								  start = zzMarkedPos; cssPrevState = zzLexicalState; yybegin(INTAG_STYLE);
								}
	"<!"						{ start = zzMarkedPos-2; yybegin(DTD); }
	"<?"						{ start = zzMarkedPos-2; yybegin(PI); }
	"<%--"					{ start = zzMarkedPos-4; yybegin(HIDDEN_COMMENT); }
	{JspStart}				{ addToken(Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }
	"<%@"                         { addToken(Token.MARKUP_TAG_DELIMITER); yybegin(JSP_DIRECTIVE); }
	"<"({Letter}|{Digit})+		{
									int count = yylength();
									addToken(zzStartRead,zzStartRead, Token.MARKUP_TAG_DELIMITER);
									zzMarkedPos -= (count-1); //yypushback(count-1);
									yybegin(INTAG_CHECK_TAG_NAME);
								}
	"</"({Letter}|{Digit})+		{
									int count = yylength();
									addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
									zzMarkedPos -= (count-2); //yypushback(count-2);
									yybegin(INTAG_CHECK_TAG_NAME);
								}
	"<"							{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(INTAG); }
	"</"						{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(INTAG); }
	{LineTerminator}			{ addNullToken(); return firstToken; }
	{Identifier}				{ addToken(Token.IDENTIFIER); } // Catches everything.
	{EntityReference}			{ addToken(Token.MARKUP_ENTITY_REFERENCE); }
	{Whitespace}+				{ addToken(Token.WHITESPACE); }
	<<EOF>>					{ addNullToken(); return firstToken; }
}

<COMMENT> {
	[^hwf\n\-]+				{}
	{URL}					{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.MARKUP_COMMENT); addHyperlinkToken(temp,zzMarkedPos-1, Token.MARKUP_COMMENT); start = zzMarkedPos; }
	{URL}					{
                                int temp = zzStartRead;
                                if (start <= zzStartRead - 1) {
                                    addToken(start,zzStartRead-1, TokenTypes.MARKUP_COMMENT);
                                }
                                addHyperlinkToken(temp,zzMarkedPos-1, TokenTypes.MARKUP_COMMENT);
                                start = zzMarkedPos;
                                start = zzMarkedPos;
                            }
	[hwf]					{}
	"-->"					{ yybegin(YYINITIAL); addToken(start,zzStartRead+2, Token.MARKUP_COMMENT); }
	"-"						{}
	{LineTerminator} |
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_COMMENT); return firstToken; }
}

<HIDDEN_COMMENT> {
	[^hwf\n\-]+				{}
	{URL}					{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.MARKUP_COMMENT); addHyperlinkToken(temp,zzMarkedPos-1, Token.MARKUP_COMMENT); start = zzMarkedPos; }
	[hwf]					{}
	"--%>"					{ yybegin(YYINITIAL); addToken(start,zzStartRead+3, Token.MARKUP_COMMENT); }
	"-"						{}
	{LineTerminator} |
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_COMMENT); addEndToken(INTERNAL_IN_HIDDEN_COMMENT); return firstToken; }
}

<PI> {
	[^\n\?]+					{}
	{LineTerminator}			{ addToken(start,zzStartRead-1, Token.MARKUP_PROCESSING_INSTRUCTION); return firstToken; }
	"?>"						{ yybegin(YYINITIAL); addToken(start,zzStartRead+1, Token.MARKUP_PROCESSING_INSTRUCTION); }
	"?"						{}
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_PROCESSING_INSTRUCTION); return firstToken; }
}

<DTD> {
	[^\n>]+					{}
	{LineTerminator}			{ addToken(start,zzStartRead-1, Token.MARKUP_DTD); return firstToken; }
	">"						{ yybegin(YYINITIAL); addToken(start,zzStartRead, Token.MARKUP_DTD); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_DTD); return firstToken; }
}

<INTAG_CHECK_TAG_NAME> {
	[Aa] |
	[aA][bB][bB][rR] |
	[aA][cC][rR][oO][nN][yY][mM] |
	[aA][dD][dD][rR][eE][sS][sS] |
	[aA][pP][pP][lL][eE][tT] |
	[aA][rR][eE][aA] |
	[aA][rR][tT][iI][cC][lL][eE] |
	[aA][sS][iI][dD][eE] |
	[aA][uU][dD][iI][oO] |
	[bB] |
	[bB][aA][sS][eE] |
	[bB][aA][sS][eE][fF][oO][nN][tT] |
	[bB][dD][oO] |
	[bB][gG][sS][oO][uU][nN][dD] |
	[bB][iI][gG] |
	[bB][lL][iI][nN][kK] |
	[bB][lL][oO][cC][kK][qQ][uU][oO][tT][eE] |
	[bB][oO][dD][yY] |
	[bB][rR] |
	[bB][uU][tT][tT][oO][nN] |
	[cC][aA][nN][vV][aA][sS] |
	[cC][aA][pP][tT][iI][oO][nN] |
	[cC][eE][nN][tT][eE][rR] |
	[cC][iI][tT][eE] |
	[cC][oO][dD][eE] |
	[cC][oO][lL] |
	[cC][oO][lL][gG][rR][oO][uU][pP] |
	[cC][oO][mM][mM][aA][nN][dD] |
	[cC][oO][mM][mM][eE][nN][tT] |
	[dD][dD] |
	[dD][aA][tT][aA][gG][rR][iI][dD] |
	[dD][aA][tT][aA][lL][iI][sS][tT] |
	[dD][aA][tT][aA][tT][eE][mM][pP][lL][aA][tT][eE] |
	[dD][eE][lL] |
	[dD][eE][tT][aA][iI][lL][sS] |
	[dD][fF][nN] |
	[dD][iI][aA][lL][oO][gG] |
	[dD][iI][rR] |
	[dD][iI][vV] |
	[dD][lL] |
	[dD][tT] |
	[eE][mM] |
	[eE][mM][bB][eE][dD] |
	[eE][vV][eE][nN][tT][sS][oO][uU][rR][cC][eE] |
	[fF][iI][eE][lL][dD][sS][eE][tT] |
	[fF][iI][gG][uU][rR][eE] |
	[fF][oO][nN][tT] |
	[fF][oO][oO][tT][eE][rR] |
	[fF][oO][rR][mM] |
	[fF][rR][aA][mM][eE] |
	[fF][rR][aA][mM][eE][sS][eE][tT] |
	[hH][123456] |
	[hH][eE][aA][dD] |
	[hH][eE][aA][dD][eE][rR] |
	[hH][rR] |
	[hH][tT][mM][lL] |
	[iI] |
	[iI][fF][rR][aA][mM][eE] |
	[iI][lL][aA][yY][eE][rR] |
	[iI][mM][gG] |
	[iI][nN][pP][uU][tT] |
	[iI][nN][sS] |
	[iI][sS][iI][nN][dD][eE][xX] |
	[kK][bB][dD] |
	[kK][eE][yY][gG][eE][nN] |
	[lL][aA][bB][eE][lL] |
	[lL][aA][yY][eE][rR] |
	[lL][eE][gG][eE][nN][dD] |
	[lL][iI] |
	[lL][iI][nN][kK] |
	[mM][aA][pP] |
	[mM][aA][rR][kK] |
	[mM][aA][rR][qQ][uU][eE][eE] |
	[mM][eE][nN][uU] |
	[mM][eE][tT][aA] |
	[mM][eE][tT][eE][rR] |
	[mM][uU][lL][tT][iI][cC][oO][lL] |
	[nN][aA][vV] |
	[nN][eE][sS][tT] |
	[nN][oO][bB][rR] |
	[nN][oO][eE][mM][bB][eE][dD] |
	[nN][oO][fF][rR][aA][mM][eE][sS] |
	[nN][oO][lL][aA][yY][eE][rR] |
	[nN][oO][sS][cC][rR][iI][pP][tT] |
	[oO][bB][jJ][eE][cC][tT] |
	[oO][lL] |
	[oO][pP][tT][gG][rR][oO][uU][pP] |
	[oO][pP][tT][iI][oO][nN] |
	[oO][uU][tT][pP][uU][tT] |
	[pP] |
	[pP][aA][rR][aA][mM] |
	[pP][lL][aA][iI][nN][tT][eE][xX][tT] |
	[pP][rR][eE] |
	[pP][rR][oO][gG][rR][eE][sS][sS] |
	[qQ] |
	[rR][uU][lL][eE] |
	[sS] |
	[sS][aA][mM][pP] |
	[sS][cC][rR][iI][pP][tT] |
	[sS][eE][cC][tT][iI][oO][nN] |
	[sS][eE][lL][eE][cC][tT] |
	[sS][eE][rR][vV][eE][rR] |
	[sS][mM][aA][lL][lL] |
	[sS][oO][uU][rR][cC][eE] |
	[sS][pP][aA][cC][eE][rR] |
	[sS][pP][aA][nN] |
	[sS][tT][rR][iI][kK][eE] |
	[sS][tT][rR][oO][nN][gG] |
	[sS][tT][yY][lL][eE] |
	[sS][uU][bB] |
	[sS][uU][pP] |
	[tT][aA][bB][lL][eE] |
	[tT][bB][oO][dD][yY] |
	[tT][dD] |
	[tT][eE][xX][tT][aA][rR][eE][aA] |
	[tT][fF][oO][oO][tT] |
	[tT][hH] |
	[tT][hH][eE][aA][dD] |
	[tT][iI][mM][eE] |
	[tT][iI][tT][lL][eE] |
	[tT][rR] |
	[tT][tT] |
	[uU] |
	[uU][lL] |
	[vV][aA][rR] |
	[vV][iI][dD][eE][oO]    { addToken(Token.MARKUP_TAG_NAME); }
	{InTagIdentifier}		{ /* A non-recognized HTML tag name */ yypushback(yylength()); yybegin(INTAG); }
	.						{ /* Shouldn't happen */ yypushback(1); yybegin(INTAG); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG); return firstToken; }
}

<INTAG> {
	{JspStart}				{ addToken(Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }
	"/"						{ addToken(Token.MARKUP_TAG_DELIMITER); }
	{InTagIdentifier}			{ addToken(Token.MARKUP_TAG_ATTRIBUTE); }
	{Whitespace}				{ addToken(Token.WHITESPACE); }
	"="						{ addToken(Token.OPERATOR); }
	"/>"						{ yybegin(YYINITIAL); addToken(Token.MARKUP_TAG_DELIMITER); }
	">"						{ yybegin(YYINITIAL); addToken(Token.MARKUP_TAG_DELIMITER); }
	[\"]						{ start = zzMarkedPos-1; yybegin(INATTR_DOUBLE); }
	[\']						{ start = zzMarkedPos-1; yybegin(INATTR_SINGLE); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG); return firstToken; }
}

<INATTR_DOUBLE> {
	{JspStart}				{ int temp=zzStartRead; if (zzStartRead>start) addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addToken(temp, zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }
	[^\"<]*					{}
	"<"						{ /* Allowing JSP expressions, etc. */ }
	[\"]					{ addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); yybegin(INTAG); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_DOUBLE); return firstToken; }
}

<INATTR_SINGLE> {
	{JspStart}				{ int temp=zzStartRead; if (zzStartRead>start) addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addToken(temp, zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }
	[^\'<]*					{}
	"<"						{ /* Allowing JSP expressions, etc. */ }
	[\']					{ addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); yybegin(INTAG); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_SINGLE); return firstToken; }
}

<INTAG_SCRIPT> {
	{JspStart}				{ addToken(Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }
	{InTagIdentifier}			{ addToken(Token.MARKUP_TAG_ATTRIBUTE); }
	"/>"					{	addToken(Token.MARKUP_TAG_DELIMITER); yybegin(YYINITIAL); }
	"/"						{ addToken(Token.MARKUP_TAG_DELIMITER); } // Won't appear in valid HTML.
	{Whitespace}+				{ addToken(Token.WHITESPACE); }
	"="						{ addToken(Token.OPERATOR); }
	">"						{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(JAVASCRIPT, LANG_INDEX_JS); }
	[\"]						{ start = zzMarkedPos-1; yybegin(INATTR_DOUBLE_SCRIPT); }
	[\']						{ start = zzMarkedPos-1; yybegin(INATTR_SINGLE_SCRIPT); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG_SCRIPT); return firstToken; }
}

<INATTR_DOUBLE_SCRIPT> {
	{JspStart}				{ int temp=zzStartRead; if (zzStartRead>start) addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addToken(temp, zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }
	[^\"<]*					{}
	"<"						{ /* Allowing JSP expressions, etc. */ }
	[\"]					{ yybegin(INTAG_SCRIPT); addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_DOUBLE_QUOTE_SCRIPT); return firstToken; }
}

<INATTR_SINGLE_SCRIPT> {
	{JspStart}				{ int temp=zzStartRead; if (zzStartRead>start) addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addToken(temp, zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }
	[^\'<]*					{}
	"<"						{ /* Allowing JSP expressions, etc. */ }
	[\']					{ yybegin(INTAG_SCRIPT); addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_SINGLE_QUOTE_SCRIPT); return firstToken; }
}

<INTAG_STYLE> {
	{InTagIdentifier}			{ addToken(Token.MARKUP_TAG_ATTRIBUTE); }
	"/>"					{	addToken(Token.MARKUP_TAG_DELIMITER); yybegin(YYINITIAL); }
	"/"						{ addToken(Token.MARKUP_TAG_DELIMITER); } // Won't appear in valid HTML.
	{Whitespace}				{ addToken(Token.WHITESPACE); }
	"="						{ addToken(Token.OPERATOR); }
	">"						{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(CSS, LANG_INDEX_CSS); }
	[\"]						{ start = zzMarkedPos-1; yybegin(INATTR_DOUBLE_STYLE); }
	[\']						{ start = zzMarkedPos-1; yybegin(INATTR_SINGLE_STYLE); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG_STYLE); return firstToken; }
}

<INATTR_DOUBLE_STYLE> {
	[^\"]*						{}
	[\"]						{ yybegin(INTAG_STYLE); addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_DOUBLE_QUOTE_STYLE); return firstToken; }
}

<INATTR_SINGLE_STYLE> {
	[^\']*						{}
	[\']						{ yybegin(INTAG_STYLE); addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_SINGLE_QUOTE_STYLE); return firstToken; }
}

<JAVASCRIPT> {

	{EndScriptTag}					{
								  yybegin(YYINITIAL, LANG_INDEX_DEFAULT);
								  addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
								  addToken(zzMarkedPos-7,zzMarkedPos-2, Token.MARKUP_TAG_NAME);
								  addToken(zzMarkedPos-1,zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER);
								}

	// ECMA 3+ keywords.
	"break" |
	"continue" |
	"delete" |
	"else" |
	"for" |
	"function" |
	"if" |
	"in" |
	"new" |
	"this" |
	"typeof" |
	"var" |
	"void" |
	"while" |
	"with"						{ addToken(Token.RESERVED_WORD); }
	"return"					{ addToken(Token.RESERVED_WORD_2); }

	//JavaScript 1.6
	"each" 						{if(JavaScriptTokenMaker.isJavaScriptCompatible("1.6")){ addToken(Token.RESERVED_WORD);} else {addToken(Token.IDENTIFIER);} }
	//JavaScript 1.7
	"let" 						{if(JavaScriptTokenMaker.isJavaScriptCompatible("1.7")){ addToken(Token.RESERVED_WORD);} else {addToken(Token.IDENTIFIER);} }

	// Reserved (but not yet used) ECMA keywords.
	"abstract" |
	"case" |
	"catch" |
	"class" |
	"const" |
	"debugger" |
	"default" |
	"do" |
	"enum" |
	"export" |
	"extends" |
	"final" |
	"finally" |
	"goto" |
	"implements" |
	"import" |
	"instanceof" |
	"interface" |
	"native" |
	"package" |
	"private" |
	"protected" |
	"public" |
	"static" |
	"super" |
	"switch" |
	"synchronized" |
	"throw" |
	"throws" |
	"transient" |
	"try" |
	"volatile" |
	"null"						{ addToken(Token.RESERVED_WORD); }
	{PrimitiveTypes}				{ addToken(Token.DATA_TYPE); }

	// Literals.
	{BooleanLiteral}				{ addToken(Token.LITERAL_BOOLEAN); }
	"NaN"						{ addToken(Token.RESERVED_WORD); }
	"Infinity"					{ addToken(Token.RESERVED_WORD); }

	// Functions.
	"eval" |
	"parseInt" |
	"parseFloat" |
	"escape" |
	"unescape" |
	"isNaN" |
	"isFinite"					{ addToken(Token.FUNCTION); }

	{LineTerminator}				{ addEndToken(INTERNAL_IN_JS); return firstToken; }
	{JS_Identifier}				{ addToken(Token.IDENTIFIER); }
	{Whitespace}+					{ addToken(Token.WHITESPACE); }

	/* String/Character literals. */
	[\']							{ start = zzMarkedPos-1; validJSString = true; yybegin(JS_CHAR); }
	[\"]							{ start = zzMarkedPos-1; validJSString = true; yybegin(JS_STRING); }
	[\`]							{ start = zzMarkedPos-1; validJSString = true; yybegin(JS_TEMPLATE_LITERAL); }

	/* Comment literals. */
	"/**/"						{ addToken(Token.COMMENT_MULTILINE); }
	{JS_MLCBegin}					{ start = zzMarkedPos-2; yybegin(JS_MLC); }
	{JS_LineCommentBegin}			{ start = zzMarkedPos-2; yybegin(JS_EOL_COMMENT); }

	/* Attempt to identify regular expressions (not foolproof) - do after comments! */
	{JS_Regex}						{
										boolean highlightedAsRegex = false;
										if (firstToken==null) {
											addToken(Token.REGEX);
											highlightedAsRegex = true;
										}
										else {
											// If this is *likely* to be a regex, based on
											// the previous token, highlight it as such.
											Token t = firstToken.getLastNonCommentNonWhitespaceToken();
											if (RSyntaxUtilities.regexCanFollowInJavaScript(t)) {
												addToken(Token.REGEX);
												highlightedAsRegex = true;
											}
										}
										// If it doesn't *appear* to be a regex, highlight it as
										// individual tokens.
										if (!highlightedAsRegex) {
											int temp = zzStartRead + 1;
											addToken(zzStartRead, zzStartRead, Token.OPERATOR);
											zzStartRead = zzCurrentPos = zzMarkedPos = temp;
										}
									}

	/* Separators. */
	{JS_Separator}					{ addToken(Token.SEPARATOR); }
	{JS_Separator2}				{ addToken(Token.IDENTIFIER); }

	{JspStart}				{ addToken(Token.MARKUP_TAG_DELIMITER); jspInState = zzLexicalState; yybegin(JAVA_EXPRESSION); }

	/* Operators. */
	{JS_Operator}					{ addToken(Token.OPERATOR); }

	/* Numbers */
	{JS_IntegerLiteral}				{ addToken(Token.LITERAL_NUMBER_DECIMAL_INT); }
	{JS_HexLiteral}				{ addToken(Token.LITERAL_NUMBER_HEXADECIMAL); }
	{JS_FloatLiteral}				{ addToken(Token.LITERAL_NUMBER_FLOAT); }
	{JS_ErrorNumberFormat}			{ addToken(Token.ERROR_NUMBER_FORMAT); }

	{JS_ErrorIdentifier}			{ addToken(Token.ERROR_IDENTIFIER); }

	/* Ended with a line not in a string or comment. */
	<<EOF>>						{ addEndToken(INTERNAL_IN_JS); return firstToken; }

	/* Catch any other (unhandled) characters and flag them as bad. */
	.							{ addToken(Token.ERROR_IDENTIFIER); }

}


<JS_STRING> {
	[^\n\\\"]+				{}
	\n						{ addToken(start,zzStartRead-1, Token.ERROR_STRING_DOUBLE); addEndToken(INTERNAL_IN_JS); return firstToken; }
	\\x{HexDigit}{2}		{}
	\\x						{ /* Invalid latin-1 character \xXX */ validJSString = false; }
	\\u{HexDigit}{4}		{}
	\\u						{ /* Invalid Unicode character \\uXXXX */ validJSString = false; }
	\\.						{ /* Skip all escaped chars. */ }
	\\						{ /* Line ending in '\' => continue to next line. */
								if (validJSString) {
									addToken(start,zzStartRead, Token.LITERAL_STRING_DOUBLE_QUOTE);
									addEndToken(INTERNAL_IN_JS_STRING_VALID);
								}
								else {
									addToken(start,zzStartRead, Token.ERROR_STRING_DOUBLE);
									addEndToken(INTERNAL_IN_JS_STRING_INVALID);
								}
								return firstToken;
							}
	\"						{ int type = validJSString ? Token.LITERAL_STRING_DOUBLE_QUOTE : Token.ERROR_STRING_DOUBLE; addToken(start,zzStartRead, type); yybegin(JAVASCRIPT); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.ERROR_STRING_DOUBLE); addEndToken(INTERNAL_IN_JS); return firstToken; }
}

<JS_CHAR> {
	[^\n\\\']+				{}
	\n						{ addToken(start,zzStartRead-1, Token.ERROR_CHAR); addEndToken(INTERNAL_IN_JS); return firstToken; }
	\\x{HexDigit}{2}		{}
	\\x						{ /* Invalid latin-1 character \xXX */ validJSString = false; }
	\\u{HexDigit}{4}		{}
	\\u						{ /* Invalid Unicode character \\uXXXX */ validJSString = false; }
	\\.						{ /* Skip all escaped chars. */ }
	\\						{ /* Line ending in '\' => continue to next line. */
								if (validJSString) {
									addToken(start,zzStartRead, Token.LITERAL_CHAR);
									addEndToken(INTERNAL_IN_JS_CHAR_VALID);
								}
								else {
									addToken(start,zzStartRead, Token.ERROR_CHAR);
									addEndToken(INTERNAL_IN_JS_CHAR_INVALID);
								}
								return firstToken;
							}
	\'						{ int type = validJSString ? Token.LITERAL_CHAR : Token.ERROR_CHAR; addToken(start,zzStartRead, type); yybegin(JAVASCRIPT); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.ERROR_CHAR); addEndToken(INTERNAL_IN_JS); return firstToken; }
}


<JS_TEMPLATE_LITERAL> {
	[^\n\\\$\`]+				{}
	\\x{HexDigit}{2}		{}
	\\x						{ /* Invalid latin-1 character \xXX */ validJSString = false; }
	\\u{HexDigit}{4}		{}
	\\u						{ /* Invalid Unicode character \\uXXXX */ validJSString = false; }
	\\.						{ /* Skip all escaped chars. */ }

	{JS_TemplateLiteralExprStart}	{
								addToken(start, zzStartRead - 1, Token.LITERAL_BACKQUOTE);
								start = zzMarkedPos-2;
								if (varDepths==null) {
									varDepths = new Stack<>();
								}
								else {
									varDepths.clear();
								}
								varDepths.push(Boolean.TRUE);
								yybegin(JS_TEMPLATE_LITERAL_EXPR);
							}
	"$"						{ /* Skip valid '$' that is not part of template literal expression start */ }
	
	\`						{ int type = validJSString ? Token.LITERAL_BACKQUOTE : Token.ERROR_STRING_DOUBLE; addToken(start,zzStartRead, type); yybegin(JAVASCRIPT); }

	/* Line ending in '\' => continue to next line, though not necessary in template strings. */
	\\						{
								if (validJSString) {
									addToken(start,zzStartRead, Token.LITERAL_BACKQUOTE);
									addEndToken(INTERNAL_IN_JS_TEMPLATE_LITERAL_VALID);
								}
								else {
									addToken(start,zzStartRead, Token.ERROR_STRING_DOUBLE);
									addEndToken(INTERNAL_IN_JS_TEMPLATE_LITERAL_INVALID);
								}
								return firstToken;
							}
	\n |
	<<EOF>>					{
								if (validJSString) {
									addToken(start, zzStartRead - 1, Token.LITERAL_BACKQUOTE);
									addEndToken(INTERNAL_IN_JS_TEMPLATE_LITERAL_VALID);
								}
								else {
									addToken(start,zzStartRead - 1, Token.ERROR_STRING_DOUBLE);
									addEndToken(INTERNAL_IN_JS_TEMPLATE_LITERAL_INVALID);
								}
								return firstToken;
							}
}

<JS_TEMPLATE_LITERAL_EXPR> {
	[^\}\$\n]+			{}
	"}"					{
							if (!varDepths.empty()) {
								varDepths.pop();
								if (varDepths.empty()) {
									addToken(start,zzStartRead, Token.VARIABLE);
									start = zzMarkedPos;
									yybegin(JS_TEMPLATE_LITERAL);
								}
							}
						}
	{JS_TemplateLiteralExprStart} { varDepths.push(Boolean.TRUE); }
	"$"					{}
	\n |
	<<EOF>>				{
							// TODO: This isn't right.  The expression and its depth should continue to the next line.
							addToken(start,zzStartRead-1, Token.VARIABLE); addEndToken(INTERNAL_IN_JS_TEMPLATE_LITERAL_INVALID); return firstToken;
						}
}

<JS_MLC> {
	// JavaScript MLC's.  This state is essentially Java's MLC state.
	[^hwf<\n\*]+				{}
	{URL}					{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addHyperlinkToken(temp,zzMarkedPos-1, Token.COMMENT_MULTILINE); start = zzMarkedPos; }
	[hwf]					{}
	{EndScriptTag}			{
							  yybegin(YYINITIAL);
							  int temp = zzStartRead;
							  addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE);
							  addToken(temp,temp+1, Token.MARKUP_TAG_DELIMITER);
							  addToken(zzMarkedPos-7,zzMarkedPos-2, Token.MARKUP_TAG_NAME);
							  addToken(zzMarkedPos-1,zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER);
							}
	"<"						{}
	\n							{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_IN_JS_MLC); return firstToken; }
	{JS_MLCEnd}					{ yybegin(JAVASCRIPT); addToken(start,zzStartRead+1, Token.COMMENT_MULTILINE); }
	\*							{}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_IN_JS_MLC); return firstToken; }
}


<JS_EOL_COMMENT> {
	[^hwf<\n]+				{}
	{URL}					{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.COMMENT_EOL); addHyperlinkToken(temp,zzMarkedPos-1, Token.COMMENT_EOL); start = zzMarkedPos; }
	[hwf]					{}
	{EndScriptTag}			{
							  int temp = zzStartRead;
							  addToken(start,zzStartRead-1, Token.COMMENT_EOL);
							  yybegin(YYINITIAL, LANG_INDEX_DEFAULT);
							  addToken(temp,temp+1, Token.MARKUP_TAG_DELIMITER);
							  addToken(zzMarkedPos-7,zzMarkedPos-2, Token.MARKUP_TAG_NAME);
							  addToken(zzMarkedPos-1,zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER);
							}
	"<"						{}
	\n						{ addToken(start,zzStartRead-1, Token.COMMENT_EOL); addEndToken(INTERNAL_IN_JS); return firstToken; }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.COMMENT_EOL); addEndToken(INTERNAL_IN_JS); return firstToken; }

}


<CSS> {
	{EndStyleTag}		{
						  yybegin(YYINITIAL, LANG_INDEX_DEFAULT);
						  addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
						  addToken(zzMarkedPos-6,zzMarkedPos-2, Token.MARKUP_TAG_NAME);
						  addToken(zzMarkedPos-1,zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER);
						}
	{CSS_SelectorPiece}	{ addToken(Token.DATA_TYPE); }
	{CSS_PseudoClass}	{ addToken(Token.RESERVED_WORD); }
	":"					{ /* Unknown pseudo class */ addToken(Token.DATA_TYPE); }
	{CSS_AtKeyword}		{ addToken(Token.REGEX); }
	{CSS_Id}			{ addToken(Token.VARIABLE); }
	"{"					{ addToken(Token.SEPARATOR); yybegin(CSS_PROPERTY); }
	[,]					{ addToken(Token.IDENTIFIER); }
	\"					{ start = zzMarkedPos-1; cssPrevState = zzLexicalState; yybegin(CSS_STRING); }
	\'					{ start = zzMarkedPos-1; cssPrevState = zzLexicalState; yybegin(CSS_CHAR_LITERAL); }
	[+>~\^$\|=]			{ addToken(Token.OPERATOR); }
	{CSS_Separator}		{ addToken(Token.SEPARATOR); }
	{Whitespace}		{ addToken(Token.WHITESPACE); }
	{CSS_MlcStart}		{ start = zzMarkedPos-2; cssPrevState = zzLexicalState; yybegin(CSS_C_STYLE_COMMENT); }
	.					{ /*System.out.println("CSS: " + yytext());*/ addToken(Token.IDENTIFIER); }
	"\n" |
	<<EOF>>				{ addEndToken(INTERNAL_CSS); return firstToken; }
}

<CSS_PROPERTY> {
	{EndStyleTag}		{
						  yybegin(YYINITIAL, LANG_INDEX_DEFAULT);
						  addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
						  addToken(zzMarkedPos-6,zzMarkedPos-2, Token.MARKUP_TAG_NAME);
						  addToken(zzMarkedPos-1,zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER);
						}
	{CSS_Property}		{ addToken(Token.RESERVED_WORD); }
	"{"					{ addToken(Token.SEPARATOR); /* helps with auto-closing curlies when editing CSS */ }
	"}"					{ addToken(Token.SEPARATOR); yybegin(CSS); }
	":"					{ addToken(Token.OPERATOR); yybegin(CSS_VALUE); }
	{Whitespace}		{ addToken(Token.WHITESPACE); }
	{CSS_MlcStart}		{ start = zzMarkedPos-2; cssPrevState = zzLexicalState; yybegin(CSS_C_STYLE_COMMENT); }
	.					{ /*System.out.println("css_property: " + yytext());*/ addToken(Token.IDENTIFIER); }
	"\n" |
	<<EOF>>				{ addEndToken(INTERNAL_CSS_PROPERTY); return firstToken; }
}

<CSS_VALUE> {
	{EndStyleTag}		{
						  yybegin(YYINITIAL, LANG_INDEX_DEFAULT);
						  addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
						  addToken(zzMarkedPos-6,zzMarkedPos-2, Token.MARKUP_TAG_NAME);
						  addToken(zzMarkedPos-1,zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER);
						}
	{CSS_Value}			{ addToken(Token.IDENTIFIER); }
	"!important"		{ addToken(Token.PREPROCESSOR); }
	{CSS_Function}		{ int temp = zzMarkedPos - 2;
						  addToken(zzStartRead, temp, Token.FUNCTION);
						  addToken(zzMarkedPos-1, zzMarkedPos-1, Token.SEPARATOR);
						  zzStartRead = zzCurrentPos = zzMarkedPos;
						}
	{CSS_Number}		{ addToken(Token.LITERAL_NUMBER_DECIMAL_INT); }
	\"					{ start = zzMarkedPos-1; cssPrevState = zzLexicalState; yybegin(CSS_STRING); }
	\'					{ start = zzMarkedPos-1; cssPrevState = zzLexicalState; yybegin(CSS_CHAR_LITERAL); }
	")"					{ /* End of a function */ addToken(Token.SEPARATOR); }
	[;]					{ addToken(Token.OPERATOR); yybegin(CSS_PROPERTY); }
	[,\.]				{ addToken(Token.IDENTIFIER); }
	"}"					{ addToken(Token.SEPARATOR); yybegin(CSS); }
	{Whitespace}		{ addToken(Token.WHITESPACE); }
	{CSS_MlcStart}		{ start = zzMarkedPos-2; cssPrevState = zzLexicalState; yybegin(CSS_C_STYLE_COMMENT); }
	.					{ /*System.out.println("css_value: " + yytext());*/ addToken(Token.IDENTIFIER); }
	"\n" |
	<<EOF>>				{ addEndToken(INTERNAL_CSS_VALUE); return firstToken; }
}

<CSS_STRING> {
	[^\n\\\"]+			{}
	\\.?				{ /* Skip escaped chars. */ }
	\"					{ addToken(start,zzStartRead, Token.LITERAL_STRING_DOUBLE_QUOTE); yybegin(cssPrevState); }
	\n |
	<<EOF>>				{ addToken(start,zzStartRead-1, Token.LITERAL_STRING_DOUBLE_QUOTE); addEndToken(INTERNAL_CSS_STRING - cssPrevState); return firstToken; }
}

<CSS_CHAR_LITERAL> {
	[^\n\\\']+			{}
	\\.?				{ /* Skip escaped chars. */ }
	\'					{ addToken(start,zzStartRead, Token.LITERAL_CHAR); yybegin(cssPrevState); }
	\n |
	<<EOF>>				{ addToken(start,zzStartRead-1, Token.LITERAL_CHAR); addEndToken(INTERNAL_CSS_CHAR - cssPrevState); return firstToken; }
}

<CSS_C_STYLE_COMMENT> {
	[^hwf\n\*]+			{}
	{URL}				{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addHyperlinkToken(temp,zzMarkedPos-1, Token.COMMENT_MULTILINE); start = zzMarkedPos; }
	[hwf]				{}
	{CSS_MlcEnd}		{ addToken(start,zzStartRead+1, Token.COMMENT_MULTILINE); yybegin(cssPrevState); }
	\*					{}
	\n |
	<<EOF>>				{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_CSS_MLC - cssPrevState); return firstToken; }
}


<JAVA_EXPRESSION> {

	"%>"							{ addToken(Token.MARKUP_TAG_DELIMITER); start = zzMarkedPos; yybegin(jspInState); }

	/* Keywords */
	"abstract"|
	"assert" |
	"break"	 |
	"case"	 |
	"catch"	 |
	"class"	 |
	"const"	 |
	"continue" |
	"default" |
	"do"	 |
	"else"	 |
	"enum"	 |
	"extends" |
	"final"	 |
	"finally" |
	"for"	 |
	"goto"	 |
	"if"	 |
	"implements" |
	"import" |
	"instanceof" |
	"interface" |
	"native" |
	"new"	 |
	"null"	 |
	"package" |
	"private" |
	"protected" |
	"public" |
	"static" |
	"strictfp" |
	"super"	 |
	"switch" |
	"synchronized" |
	"this"	 |
	"throw"	 |
	"throws" |
	"transient" |
	"try"	 |
	"void"	 |
	"volatile" |
	"while"					{ addToken(Token.RESERVED_WORD); }
	"return"				{ addToken(Token.RESERVED_WORD_2); }

	/* Data types. */
	{PrimitiveTypes}			{ addToken(Token.DATA_TYPE); }

	/* Booleans. */
	{BooleanLiteral}			{ addToken(Token.LITERAL_BOOLEAN); }

	/* java.lang classes */
	"Appendable" |
	"AutoCloseable" |
	"CharSequence" |
	"Cloneable" |
	"Comparable" |
	"Iterable" |
	"Readable" |
	"Runnable" |
	"Thread.UncaughtExceptionHandler" |
	"Boolean" |
	"Byte" |
	"Character" |
	"Character.Subset" | 
	"Character.UnicodeBlock" | 
	"Class" |
	"ClassLoader" |
	"ClassValue" |
	"Compiler" |
	"Double" |
	"Enum" |
	"Float" |
	"InheritableThreadLocal" |
	"Integer" |
	"Long" |
	"Math" |
	"Number" |
	"Object" |
	"Package" |
	"Process" |
	"ProcessBuilder" |
	"ProcessBuilder.Redirect" |
	"Runtime" |
	"RuntimePermission" |
	"SecurityManager" |
	"Short" |
	"StackTraceElement" |
	"StrictMath" |
	"String" |
	"StringBuffer" |
	"StringBuilder" |
	"System" |
	"Thread" |
	"ThreadGroup" |
	"ThreadLocal" |
	"Throwable" |
	"Void" |
	"Character.UnicodeScript" |
	"ProcessBuilder.Redirect.Type" |
	"Thread.State" |
	"ArithmeticException" |
	"ArrayIndexOutOfBoundsException" |
	"ArrayStoreException" |
	"ClassCastException" |
	"ClassNotFoundException" |
	"CloneNotSupportedException" |
	"EnumConstantNotPresentException" |
	"Exception" |
	"IllegalAccessException" |
	"IllegalArgumentException" |
	"IllegalMonitorStateException" |
	"IllegalStateException" |
	"IllegalThreadStateException" |
	"IndexOutOfBoundsException" |
	"InstantiationException" |
	"InterruptedException" |
	"NegativeArraySizeException" |
	"NoSuchFieldException" |
	"NoSuchMethodException" |
	"NullPointerException" |
	"NumberFormatException" |
	"RuntimeException" |
	"SecurityException" |
	"StringIndexOutOfBoundsException" |
	"TypeNotPresentException" |
	"UnsupportedOperationException" |
	"AbstractMethodError" |
	"AssertionError" |
	"BootstrapMethodError" |
	"ClassCircularityError" |
	"ClassFormatError" |
	"Error" |
	"ExceptionInInitializerError" |
	"IllegalAccessError" |
	"IncompatibleClassChangeError" |
	"InstantiationError" |
	"InternalError" |
	"LinkageError" |
	"NoClassDefFoundError" |
	"NoSuchFieldError" |
	"NoSuchMethodError" |
	"OutOfMemoryError" |
	"StackOverflowError" |
	"ThreadDeath" |
	"UnknownError" |
	"UnsatisfiedLinkError" |
	"UnsupportedClassVersionError" |
	"VerifyError" |
	"VirtualMachineError" |

	/* java.io classes*/
    "Closeable" |
    "DataInput" |
    "DataOutput" |
    "Externalizable" |
    "FileFilter" |
    "FilenameFilter" |
    "Flushable" |
    "ObjectInput" |
    "ObjectInputValidation" |
    "ObjectOutput" |
    "ObjectStreamConstants" |
    "Serializable" |

    "BufferedInputStream" |
    "BufferedOutputStream" |
    "BufferedReader" |
    "BufferedWriter" |
    "ByteArrayInputStream" |
    "ByteArrayOutputStream" |
    "CharArrayReader" |
    "CharArrayWriter" |
    "Console" |
    "DataInputStream" |
    "DataOutputStream" |
    "File" |
    "FileDescriptor" |
    "FileInputStream" |
    "FileOutputStream" |
    "FilePermission" |
    "FileReader" |
    "FileWriter" |
    "FilterInputStream" |
    "FilterOutputStream" |
    "FilterReader" |
    "FilterWriter" |
    "InputStream" |
    "InputStreamReader" |
    "LineNumberInputStream" |
    "LineNumberReader" |
    "ObjectInputStream" |
    "ObjectInputStream.GetField" |
    "ObjectOutputStream" |
    "ObjectOutputStream.PutField" |
    "ObjectStreamClass" |
    "ObjectStreamField" |
    "OutputStream" |
    "OutputStreamWriter" |
    "PipedInputStream" |
    "PipedOutputStream" |
    "PipedReader" |
    "PipedWriter" |
    "PrintStream" |
    "PrintWriter" |
    "PushbackInputStream" |
    "PushbackReader" |
    "RandomAccessFile" |
    "Reader" |
    "SequenceInputStream" |
    "SerializablePermission" |
    "StreamTokenizer" |
    "StringBufferInputStream" |
    "StringReader" |
    "StringWriter" |
    "Writer" |

    "CharConversionException" |
    "EOFException" |
    "FileNotFoundException" |
    "InterruptedIOException" |
    "InvalidClassException" |
    "InvalidObjectException" |
    "IOException" |
    "NotActiveException" |
    "NotSerializableException" |
    "ObjectStreamException" |
    "OptionalDataException" |
    "StreamCorruptedException" |
    "SyncFailedException" |
    "UncheckedIOException" |
    "UnsupportedEncodingException" |
    "UTFDataFormatException" |
    "WriteAbortedException" |

    "IOError" |

	/* java.util classes */
    "Collection" |
    "Comparator" |
    "Deque" |
    "Enumeration" |
    "EventListener" |
    "Formattable" |
    "Iterator" |
    "List" |
    "ListIterator" |
    "Map" |
    "Map.Entry" |
    "NavigableMap" |
    "NavigableSet" |
    "Observer" |
    "PrimitiveIterator" |
    "PrimitiveIterator.OfDouble" |
    "PrimitiveIterator.OfInt" |
    "PrimitiveIterator.OfLong" |
    "Queue" |
    "RandomAccess" |
    "Set" |
    "SortedMap" |
    "SortedSet" |
    "Spliterator" |
    "Spliterator.OfDouble" |
    "Spliterator.OfInt" |
    "Spliterator.OfLong" |
    "Spliterator.OfPrimitive" |

    "AbstractCollection" |
    "AbstractList" |
    "AbstractMap" |
    "AbstractMap.SimpleEntry" |
    "AbstractMap.SimpleImmutableEntry" |
    "AbstractQueue" |
    "AbstractSequentialList" |
    "AbstractSet" |
    "ArrayDeque" |
    "ArrayList" |
    "Arrays" |
    "Base64" |
    "Base64.Decoder" |
    "Base64.Encoder" |
    "BitSet" |
    "Calendar" |
    "Calendar.Builder" |
    "Collections" |
    "Currency" |
    "Date" |
    "Dictionary" |
    "DoubleSummaryStatistics" |
    "EnumMap" |
    "EnumSet" |
    "EventListenerProxy" |
    "EventObject" |
    "FormattableFlags" |
    "Formatter" |
    "GregorianCalendar" |
    "HashMap" |
    "HashSet" |
    "Hashtable" |
    "IdentityHashMap" |
    "IntSummaryStatistics" |
    "LinkedHashMap" |
    "LinkedHashSet" |
    "LinkedList" |
    "ListResourceBundle" |
    "Locale" |
    "Locale.Builder" |
    "Locale.LanguageRange" |
    "LongSummaryStatistics" |
    "Objects" |
    "Observable" |
    "Optional" |
    "OptionalDouble" |
    "OptionalInt" |
    "OptionalLong" |
    "PriorityQueue" |
    "Properties" |
    "PropertyPermission" |
    "PropertyResourceBundle" |
    "Random" |
    "ResourceBundle" |
    "ResourceBundle.Control" |
    "Scanner" |
    "ServiceLoader" |
    "SimpleTimeZone" |
    "Spliterators" |
    "Spliterators.AbstractDoubleSpliterator" |
    "Spliterators.AbstractIntSpliterator" |
    "Spliterators.AbstractLongSpliterator" |
    "Spliterators.AbstractSpliterator" |
    "SpliteratorRandom" |
    "Stack" |
    "StringJoiner" |
    "StringTokenizer" |
    "Timer" |
    "TimerTask" |
    "TimeZone" |
    "TreeMap" |
    "TreeSet" |
    "UUID" |
    "Vector" |
    "WeakHashMap" |

    "Formatter.BigDecimalLayoutForm" |
    "Locale.Category" |
    "Locale.FilteringMode" |

    "ConcurrentModificationException" |
    "DuplicateFormatFlagsException" |
    "EmptyStackException" |
    "FormatFlagsConversionMismatchException" |
    "FormatterClosedException" |
    "IllegalFormatCodePointException" |
    "IllegalFormatConversionException" |
    "IllegalFormatException" |
    "IllegalFormatFlagsException" |
    "IllegalFormatPrecisionException" |
    "IllegalFormatWidthException" |
    "IllformedLocaleException" |
    "InputMismatchException" |
    "InvalidPropertiesFormatException" |
    "MissingFormatArgumentException" |
    "MissingFormatWidthException" |
    "MissingResourceException" |
    "NoSuchElementException" |
    "TooManyListenersException" |
    "UnknownFormatConversionException" |
    "UnknownFormatFlagsException" |

    "ServiceConfigurationError" 		{ addToken(Token.FUNCTION); }

	{LineTerminator}				{ addEndToken(INTERNAL_IN_JAVA_EXPRESSION - jspInState); return firstToken; }

	{JIdentifier}					{ addToken(Token.IDENTIFIER); }

	{WhiteSpace}+					{ addToken(Token.WHITESPACE); }

	/* String/Character literals. */
	{JCharLiteral}					{ addToken(Token.LITERAL_CHAR); }
	{JUnclosedCharLiteral}			{ addToken(Token.ERROR_CHAR); addEndToken(INTERNAL_IN_JAVA_EXPRESSION - jspInState); return firstToken; }
	{JErrorCharLiteral}				{ addToken(Token.ERROR_CHAR); }
	{JStringLiteral}				{ addToken(Token.LITERAL_STRING_DOUBLE_QUOTE); }
	{JUnclosedStringLiteral}			{ addToken(Token.ERROR_STRING_DOUBLE); addEndToken(INTERNAL_IN_JAVA_EXPRESSION - jspInState); return firstToken; }
	{JErrorStringLiteral}			{ addToken(Token.ERROR_STRING_DOUBLE); }

	/* Comment literals. */
	"/**/"						{ addToken(Token.COMMENT_MULTILINE); }
	{MLCBegin}					{ start = zzMarkedPos-2; yybegin(JAVA_MLC); }
	{DocCommentBegin}				{ start = zzMarkedPos-3; yybegin(JAVA_DOCCOMMENT); }
	{LineCommentBegin}.*			{ addToken(Token.COMMENT_EOL); addEndToken(INTERNAL_IN_JAVA_EXPRESSION - jspInState); return firstToken; }

	/* Annotations. */
	{Annotation}					{ addToken(Token.ANNOTATION); }

	/* Separators. */
	{Separator}					{ addToken(Token.SEPARATOR); }
	{Separator2}					{ addToken(Token.IDENTIFIER); }

	/* Operators. */
	{Operator}					{ addToken(Token.OPERATOR); }

	/* Numbers */
	{IntegerLiteral}				{ addToken(Token.LITERAL_NUMBER_DECIMAL_INT); }
	{BinaryLiteral}					{ addToken(Token.LITERAL_NUMBER_DECIMAL_INT); }
	{HexLiteral}					{ addToken(Token.LITERAL_NUMBER_HEXADECIMAL); }
	{FloatLiteral}					{ addToken(Token.LITERAL_NUMBER_FLOAT); }
	{ErrorNumberFormat}				{ addToken(Token.ERROR_NUMBER_FORMAT); }

	{ErrorIdentifier}				{ addToken(Token.ERROR_IDENTIFIER); }

	/* Ended with a line not in a string or comment. */
	<<EOF>>						{ addEndToken(INTERNAL_IN_JAVA_EXPRESSION - jspInState); return firstToken; }

	/* Catch any other (unhandled) characters and flag them as bad. */
	.							{ addToken(Token.ERROR_IDENTIFIER); }

}


<JAVA_MLC> {
	[^hwf\n\*]+				{}
	{URL}					{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addHyperlinkToken(temp,zzMarkedPos-1, Token.COMMENT_MULTILINE); start = zzMarkedPos; }
	[hwf]					{}
	{MLCEnd}					{ yybegin(JAVA_EXPRESSION); addToken(start,zzStartRead+1, Token.COMMENT_MULTILINE); }
	\*						{}
	\n |
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_IN_JAVA_MLC - jspInState); return firstToken; }
}


<JAVA_DOCCOMMENT> {

	[^hwf\@\{\n\<\*]+			{}
	{URL}						{
                                    int temp=zzStartRead;
                                    if (start <= zzStartRead - 1) {
                                        addToken(start,zzStartRead-1, Token.COMMENT_DOCUMENTATION);
                                    }
                                    addHyperlinkToken(temp,zzMarkedPos-1, Token.COMMENT_DOCUMENTATION);
                                    start = zzMarkedPos;
                                }
	[hwf]						{}

	"@"{BlockTag}				{
                                    int temp=zzStartRead;
                                    if (start <= zzStartRead - 1) {
                                        addToken(start,zzStartRead-1, Token.COMMENT_DOCUMENTATION);
                                    }
                                    addToken(temp,zzMarkedPos-1, Token.COMMENT_KEYWORD);
                                    start = zzMarkedPos; }
	"@"							{}
	"{@"{InlineTag}[^\}]*"}"	{
                                    int temp=zzStartRead;
                                    if (start <= zzStartRead - 1) {
                                        addToken(start,zzStartRead-1, Token.COMMENT_DOCUMENTATION);
                                    }
                                    addToken(temp,zzMarkedPos-1, Token.COMMENT_KEYWORD);
                                    start = zzMarkedPos;
                                }
	"{"							{}
	\n							{ addToken(start,zzStartRead-1, Token.COMMENT_DOCUMENTATION); addEndToken(INTERNAL_IN_JAVA_DOCCOMMENT - jspInState); return firstToken; }
	"<"[/]?({Letter}[^\>]*)?">"	{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.COMMENT_DOCUMENTATION); addToken(temp,zzMarkedPos-1, Token.COMMENT_MARKUP); start = zzMarkedPos; }
	\<							{}
	{MLCEnd}					{ yybegin(JAVA_EXPRESSION); addToken(start,zzStartRead+1, Token.COMMENT_DOCUMENTATION); }
	\*							{}
	<<EOF>>						{ yybegin(JAVA_EXPRESSION); addToken(start,zzEndRead, Token.COMMENT_DOCUMENTATION); addEndToken(INTERNAL_IN_JAVA_DOCCOMMENT - jspInState); return firstToken; }

}


<JSP_DIRECTIVE> {
	"include" |
	"page" |
	"taglib"					{ addToken(Token.RESERVED_WORD); }
	"/"						{ addToken(Token.RESERVED_WORD); }
	{InTagIdentifier}			{ addToken(Token.IDENTIFIER); }
	{Whitespace}+				{ addToken(Token.WHITESPACE); }
	"="						{ addToken(Token.OPERATOR); }
	"%>"						{ yybegin(YYINITIAL); addToken(Token.MARKUP_TAG_DELIMITER); }
	"%"						{ addToken(Token.IDENTIFIER); }
	">"						{ addToken(Token.IDENTIFIER); /* Needed as InTagIdentifier ignores it. */ }
	{UnclosedStringLiteral}		{ addToken(Token.ERROR_STRING_DOUBLE); }
	{StringLiteral}			{ addToken(Token.LITERAL_STRING_DOUBLE_QUOTE); }
	{UnclosedCharLiteral}		{ addToken(Token.ERROR_CHAR); }
	{CharLiteral}				{ addToken(Token.LITERAL_CHAR); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_IN_JSP_DIRECTIVE); return firstToken; }
}

