/*
 * MarkdownTokenMaker.java - Generates tokens for Markdown syntax highlighting.
 * 
 * This library is distributed under a modified BSD license.  See the included
 * LICENSE file for details.
 */
package org.fife.ui.rsyntaxtextarea.modes;

import java.io.*;
import javax.swing.text.Segment;

import org.fife.ui.rsyntaxtextarea.*;


/**
 * Scanner for Markdown files.  This is based off of our {@code HTMLTokenMaker}
 * class, and as such, probably highlights more HTML than it should.  Still, it
 * should be good enough for most use cases.
 *
 * This implementation was created using
 * <a href="https://www.jflex.de/">JFlex</a> 1.4.1; however, the generated file
 * was modified for performance.  Memory allocation needs to be almost
 * completely removed to be competitive with the handwritten lexers (subclasses
 * of <code>AbstractTokenMaker</code>), so this class has been modified so that
 * Strings are never allocated (via yytext()), and the scanner never has to
 * worry about refilling its buffer (needlessly copying chars around).
 * We can achieve this because RText always scans exactly 1 line of tokens at a
 * time, and hands the scanner this line as an array of characters (a Segment
 * really).  Since tokens contain pointers to char arrays instead of Strings
 * holding their contents, there is no need for allocating new memory for
 * Strings.<p>
 *
 * The actual algorithm generated for scanning has, of course, not been
 * modified.<p>
 *
 * If you wish to regenerate this file yourself, keep in mind the following:
 * <ul>
 *   <li>The generated <code>MarkdownTokenMaker.java</code> file will contain two
 *       definitions of both <code>zzRefill</code> and <code>yyreset</code>.
 *       You should hand-delete the second of each definition (the ones
 *       generated by the lexer), as these generated methods modify the input
 *       buffer, which we'll never have to do.</li>
 *   <li>You should also change the declaration/definition of zzBuffer to NOT
 *       be initialized.  This is a needless memory allocation for us since we
 *       will be pointing the array somewhere else anyway.</li>
 *   <li>You should NOT call <code>yylex()</code> on the generated scanner
 *       directly; rather, you should use <code>getTokenList</code> as you would
 *       with any other <code>TokenMaker</code> instance.</li>
 * </ul>
 *
 * @author Robert Futrell
 * @version 0.9
 */
%%

%public
%class MarkdownTokenMaker
%extends AbstractMarkupTokenMaker
%unicode
%type org.fife.ui.rsyntaxtextarea.Token


%{

	/**
	 * Type specific to XMLTokenMaker denoting a line ending with an unclosed
	 * double-quote attribute.
	 */
	public static final int INTERNAL_ATTR_DOUBLE			= -1;


	/**
	 * Type specific to XMLTokenMaker denoting a line ending with an unclosed
	 * single-quote attribute.
	 */
	public static final int INTERNAL_ATTR_SINGLE			= -2;


	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed HTML tag; thus a new line is beginning
	 * still inside the tag.
	 */
	public static final int INTERNAL_INTAG					= -3;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "***" bold/italic tag; thus a new line is
	 * beginning still inside the tag.
	 */
    public static final int INTERNAL_IN_BOLDITALIC1         = -4;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "___" bold/italic tag; thus a new line is
	 * beginning still inside the tag.
	 */
    public static final int INTERNAL_IN_BOLDITALIC2         = -5;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "**" bold tag; thus a new line is
	 * beginning still inside the tag.
	 */
    public static final int INTERNAL_IN_BOLD1         = -6;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "__" bold tag; thus a new line is
	 * beginning still inside the tag.
	 */
    public static final int INTERNAL_IN_BOLD2         = -7;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "*" italic tag; thus a new line is
	 * beginning still inside the tag.
	 */
    public static final int INTERNAL_IN_ITALIC1         = -8;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "_" italic tag; thus a new line is
	 * beginning still inside the tag.
	 */
    public static final int INTERNAL_IN_ITALIC2         = -9;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "`" code tag; thus a new line is
	 * beginning still inside the tag.
	 */
    public static final int INTERNAL_IN_CODE         = -10;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "```" code block; thus a new line is
	 * beginning still inside the block.
	 */
    public static final int INTERNAL_IN_SYNTAX_HIGHLIGHTING     = -11;

	/**
	 * Token type specific to MarkdownTokenMaker; this signals that the user has
	 * ended a line with an unclosed "~~" strikethrough section; thus a new line
	 * is beginning still inside the block.
	 */
    public static final int INTERNAL_IN_STRIKETHROUGH     = -12;

	/**
	 * Whether closing markup tags are automatically completed for Markdown.
	 */
	private static boolean completeCloseTags;


	/**
	 * Constructor.  This must be here because JFlex does not generate a
	 * no-parameter constructor.
	 */
	public MarkdownTokenMaker() {
		super();
	}


	/**
	 * Adds the token specified to the current linked list of tokens as an
	 * "end token;" that is, at <code>zzMarkedPos</code>.
	 *
	 * @param tokenType The token's type.
	 */
	private void addEndToken(int tokenType) {
		addToken(zzMarkedPos,zzMarkedPos, tokenType);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param tokenType The token's type.
	 */
	private void addToken(int tokenType) {
		addToken(zzStartRead, zzMarkedPos-1, tokenType);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param tokenType The token's type.
	 */
	private void addToken(int start, int end, int tokenType) {
		int so = start + offsetShift;
		addToken(zzBuffer, start,end, tokenType, so);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param array The character array.
	 * @param start The starting offset in the array.
	 * @param end The ending offset in the array.
	 * @param tokenType The token's type.
	 * @param startOffset The offset in the document at which this token
	 *                    occurs.
	 */
	@Override
	public void addToken(char[] array, int start, int end, int tokenType, int startOffset) {
		super.addToken(array, start,end, tokenType, startOffset);
		zzStartRead = zzMarkedPos;
	}


	@Override
	protected OccurrenceMarker createOccurrenceMarker() {
		return new HtmlOccurrenceMarker();
	}


	/**
	 * Sets whether markup close tags should be completed.  You might not want
	 * this to be the case, since some tags in standard HTML aren't usually
	 * closed.
	 *
	 * @return Whether closing markup tags are completed.
	 * @see #setCompleteCloseTags(boolean)
	 */
	@Override
	public boolean getCompleteCloseTags() {
		return completeCloseTags;
	}


	@Override
	public boolean getCurlyBracesDenoteCodeBlocks(int languageIndex) {
		return false;
	}


	@Override
	public String[] getLineCommentStartAndEnd(int languageIndex) {
		return null;
	}


	/**
	 * Returns <code>TokenTypes.MARKUP_TAG_NAME</code>.
	 *
	 * @param type The token type.
	 * @return Whether tokens of this type should have "mark occurrences"
	 *         enabled.
	 */
	@Override
	public boolean getMarkOccurrencesOfTokenType(int type) {
		return type==TokenTypes.MARKUP_TAG_NAME;
	}


	/**
	 * Returns the first token in the linked list of tokens generated
	 * from <code>text</code>.  This method must be implemented by
	 * subclasses so they can correctly implement syntax highlighting.
	 *
	 * @param text The text from which to get tokens.
	 * @param initialTokenType The token type we should start with.
	 * @param startOffset The offset into the document at which
	 *        <code>text</code> starts.
	 * @return The first <code>Token</code> in a linked list representing
	 *         the syntax highlighted text.
	 */
	@Override
	public Token getTokenList(Segment text, int initialTokenType, int startOffset) {

		resetTokenList();
		this.offsetShift = -text.offset + startOffset;
		int languageIndex = 0;

		// Start off in the proper state.
		int state;
		switch (initialTokenType) {
			case INTERNAL_INTAG:
				state = INTAG;
				break;
			case INTERNAL_ATTR_DOUBLE:
				state = INATTR_DOUBLE;
				break;
			case INTERNAL_ATTR_SINGLE:
				state = INATTR_SINGLE;
				break;
            case INTERNAL_IN_BOLDITALIC1:
                state = BOLDITALIC1;
                break;
            case INTERNAL_IN_BOLDITALIC2:
                state = BOLDITALIC2;
                break;
            case INTERNAL_IN_BOLD1:
                state = BOLD1;
                break;
            case INTERNAL_IN_BOLD2:
                state = BOLD2;
                break;
            case INTERNAL_IN_ITALIC1:
                state = ITALIC1;
                break;
            case INTERNAL_IN_ITALIC2:
                state = ITALIC2;
                break;
            case INTERNAL_IN_CODE:
                state = CODE;
                break;
            case INTERNAL_IN_SYNTAX_HIGHLIGHTING:
                state = SYNTAX_HIGHLIGHTING;
                break;
            case INTERNAL_IN_STRIKETHROUGH:
                state = STRIKETHROUGH;
                break;
			default:
                state = YYINITIAL;
				break;
		}

		setLanguageIndex(languageIndex);
		start = text.offset;
		s = text;
		try {
			yyreset(zzReader);
			yybegin(state);
			return yylex();
		} catch (IOException ioe) {
			ioe.printStackTrace();
			return new TokenImpl();
		}

	}


	/**
	 * Overridden to accept letters, digits, underscores, and hyphens.
	 */
	@Override
	public boolean isIdentifierChar(int languageIndex, char ch) {
        return Character.isLetterOrDigit(ch) || ch=='-' || ch=='.' || ch=='_';
	}


	/**
	 * Sets whether markup close tags should be completed.  You might not want
	 * this to be the case, since some tags in standard HTML aren't usually
	 * closed.
	 *
	 * @param complete Whether closing markup tags are completed.
	 * @see #getCompleteCloseTags()
	 */
	public static void setCompleteCloseTags(boolean complete) {
		completeCloseTags = complete;
	}


	/**
	 * Refills the input buffer.
	 *
	 * @return      <code>true</code> if EOF was reached, otherwise
	 *              <code>false</code>.
	 */
	private boolean zzRefill() {
		return zzCurrentPos>=s.offset+s.count;
	}


	/**
	 * Resets the scanner to read from a new input stream.
	 * Does not close the old reader.
	 *
	 * All internal variables are reset, the old input stream 
	 * <b>cannot</b> be reused (internal buffer is discarded and lost).
	 * Lexical state is set to <tt>YY_INITIAL</tt>.
	 *
	 * @param reader   the new input stream 
	 */
	public final void yyreset(java.io.Reader reader) {
		// 's' has been updated.
		zzBuffer = s.array;
		/*
		 * We replaced the line below with the two below it because zzRefill
		 * no longer "refills" the buffer (since the way we do it, it's always
		 * "full" the first time through, since it points to the segment's
		 * array).  So, we assign zzEndRead here.
		 */
		//zzStartRead = zzEndRead = s.offset;
		zzStartRead = s.offset;
		zzEndRead = zzStartRead + s.count - 1;
		zzCurrentPos = zzMarkedPos = zzPushbackPos = s.offset;
		zzLexicalState = YYINITIAL;
		zzReader = reader;
		zzAtBOL  = true;
		zzAtEOF  = false;
	}


%}

// HTML-specific stuff.
Whitespace					= ([ \t\f]+)
LineTerminator				= ([\n])
Identifier					= ([^ \t\n<&*`_~][^ \t\n<&*`~]*)
EntityReference					= ([&][^; \t]*[;]?)
InTagIdentifier				= ([^ \t\n\"\'/=>]+)

// Markdown stuff
HeadingStart                = [#]

// General stuff.
Letter						= [A-Za-z]
NonzeroDigit				= [1-9]
Digit						= ("0"|{NonzeroDigit})
LetterOrUnderscore			= ({Letter}|[_])


URLGenDelim				= ([:\/\?#\[\]@])
URLSubDelim				= ([\!\$&'\(\)\*\+,;=])
URLUnreserved			= ({LetterOrUnderscore}|{Digit}|[\-\.\~])
URLCharacter			= ({URLGenDelim}|{URLSubDelim}|{URLUnreserved}|[%])
URLCharacters			= ({URLCharacter}*)
URLEndCharacter			= ([\/\$]|{Letter}|{Digit})
URL						= (((https?|f(tp|ile))"://"|"www.")({URLCharacters}{URLEndCharacter})?)

EmailAddress            = ([^@]+@[^@]+\.[^@]+)

%state INTAG
%state INTAG_CHECK_TAG_NAME
%state INATTR_DOUBLE
%state INATTR_SINGLE
%state BOLDITALIC1
%state BOLDITALIC2
%state BOLD1
%state BOLD2
%state ITALIC1
%state ITALIC2
%state STRIKETHROUGH
%state CODE
%state SYNTAX_HIGHLIGHTING

%%

<YYINITIAL> {

    {HeadingStart}.*            {
                                    // In almost all cases, '#' is the beginning of a heading
                                    if (getNoTokensIdentifiedYet()) {
                                        addToken(Token.RESERVED_WORD);
                                    }
                                    // If for some reason it isn't, highlight it as an identifier and continue on
                                    else {
                                        int count = yylength();
                                        addToken(zzStartRead, zzStartRead + 1, Token.IDENTIFIER);
                                        zzMarkedPos -= (count - 1);
                                    }
                                }
    "```"                       { start = zzMarkedPos - 3; yybegin(SYNTAX_HIGHLIGHTING); }
    "```"[A-Za-z0-9-]+          {
                                    // A code block with a language specified
                                    start = zzStartRead;
                                    int length = yylength();
                                    addToken(start, start + 2, TokenTypes.PREPROCESSOR);
                                    addToken(start + 3, start + length - 1, TokenTypes.VARIABLE);
                                    start = zzMarkedPos;
                                    yybegin(SYNTAX_HIGHLIGHTING);
                                }
    [-*+]" "                    {
                                    // An unordered list item.  Trailing space is required
                                    if (TokenUtils.isBlankOrAllWhiteSpace(firstToken)) {
                                        start = zzStartRead;
                                        addToken(start, start, TokenTypes.LITERAL_NUMBER_DECIMAL_INT);
                                        addToken(start + 1, start + 1, TokenTypes.WHITESPACE);
                                    }
                                    // Something before it - not an unordered list item
                                    else {
                                        start = zzStartRead;
                                        addToken(start, start, TokenTypes.IDENTIFIER);
                                        addToken(start + 1, start + 1, TokenTypes.WHITESPACE);
                                    }
                                }
    [0-9]+". "                  {
                                    // An ordered list item.  Trailing space is required
                                    if (TokenUtils.isBlankOrAllWhiteSpace(firstToken)) {
                                        start = zzStartRead;
                                        int length = yylength();
                                        addToken(start, start + length - 3, TokenTypes.LITERAL_NUMBER_DECIMAL_INT);
                                        addToken(start + length - 2, start + length - 2, TokenTypes.IDENTIFIER);
                                        addToken(start + length - 1, start + length - 1, TokenTypes.WHITESPACE);
                                    }
                                    // Not an ordered list item - highlight the digits as plain text
                                    else {
                                        start = zzStartRead;
                                        int length = yylength();
                                        addToken(start, start + length - 2, TokenTypes.IDENTIFIER);
                                        addToken(start + length - 1, start + length - 1, TokenTypes.WHITESPACE);
                                    }
                                }
    "***"                       { start = zzMarkedPos - 3; yybegin(BOLDITALIC1); }
    "___"                       { start = zzMarkedPos - 3; yybegin(BOLDITALIC2); }
    "**"                        { start = zzMarkedPos - 2; yybegin(BOLD1); }
    "__"                        { start = zzMarkedPos - 2; yybegin(BOLD2); }
    "*"                         { start = zzMarkedPos - 1; yybegin(ITALIC1); }
    "_"                         { start = zzMarkedPos - 1; yybegin(ITALIC2); }
    [`]                         { start = zzMarkedPos - 1; yybegin(CODE); }
    "~~"                        { start = zzMarkedPos - 2; yybegin(STRIKETHROUGH); }
    "~"                         { addToken(TokenTypes.OPERATOR); }
    ">".*                       {
                                    // In many cases, '>' is the beginning of a block quote
                                    if (TokenUtils.isBlankOrAllWhiteSpace(firstToken)) {
                                        addToken(Token.COMMENT_EOL);
                                    }
                                    // But if not, highlight it as an identifier and continue on
                                    else {
                                        int count = yylength();
                                        addToken(zzStartRead, zzStartRead, Token.IDENTIFIER);
                                        zzMarkedPos -= (count - 1);
                                    }
                                }
    [!]?"["[^\]]+"]("[^\)]*")"  {
                                    // Image or URL - alt or link text + URL, optional title
                                    start = zzStartRead;
                                    int count = yylength();
                                    int rBracket = yytext().indexOf(']');
                                    addToken(start, start + rBracket, TokenTypes.REGEX);
                                    addToken(start + rBracket + 1, start + count - 1, TokenTypes.ANNOTATION);
                                }
    "<"{URL}">"                 { addToken(TokenTypes.OPERATOR); /* Quick URL syntax */ }
    "<"{EmailAddress}">"        { addToken(TokenTypes.OPERATOR); /* Quick email address syntax */ }
    "---".*                     {
                                    // An "hr" line must be "---" all by itself on a line
                                    boolean firstOnLine = getNoTokensIdentifiedYet();
                                    start = zzStartRead;
                                    if (firstOnLine) {
                                        if (yylength() == 3) {
                                            addToken(TokenTypes.COMMENT_DOCUMENTATION);
                                        }
                                        else {
                                            // "hr" markup with following content is invalid
                                            addToken(start, start + 2, Token.IDENTIFIER);
                                            zzMarkedPos = start + 3;
                                        }
                                    }
                                    else {
                                        // The "hr" markup is really just an identifier
                                        addToken(start, start + 2, Token.IDENTIFIER);
                                        zzMarkedPos = start + 3;
                                    }
                                }

	"<"({Letter}|{Digit})+		{
									int count = yylength();
									addToken(zzStartRead,zzStartRead, Token.MARKUP_TAG_DELIMITER);
									zzMarkedPos -= (count-1); //yypushback(count-1);
									yybegin(INTAG_CHECK_TAG_NAME);
								}
	"</"({Letter}|{Digit})+		{
									int count = yylength();
									addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
									zzMarkedPos -= (count-2); //yypushback(count-2);
									yybegin(INTAG_CHECK_TAG_NAME);
								}
	"<"							{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(INTAG); }
	"</"						{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(INTAG); }
	{LineTerminator}			{ addNullToken(); return firstToken; }
	{Identifier}				{ addToken(Token.IDENTIFIER); } // Catches everything.
	{EntityReference}			{ addToken(Token.MARKUP_ENTITY_REFERENCE); }
	{Whitespace}				{ addToken(Token.WHITESPACE); }
	<<EOF>>					{ addNullToken(); return firstToken; }
}

<INTAG_CHECK_TAG_NAME> {
	[Aa] |
	[aA][bB][bB][rR] |
	[aA][cC][rR][oO][nN][yY][mM] |
	[aA][dD][dD][rR][eE][sS][sS] |
	[aA][pP][pP][lL][eE][tT] |
	[aA][rR][eE][aA] |
	[aA][rR][tT][iI][cC][lL][eE] |
	[aA][sS][iI][dD][eE] |
	[aA][uU][dD][iI][oO] |
	[bB] |
	[bB][aA][sS][eE] |
	[bB][aA][sS][eE][fF][oO][nN][tT] |
	[bB][dD][oO] |
	[bB][gG][sS][oO][uU][nN][dD] |
	[bB][iI][gG] |
	[bB][lL][iI][nN][kK] |
	[bB][lL][oO][cC][kK][qQ][uU][oO][tT][eE] |
	[bB][oO][dD][yY] |
	[bB][rR] |
	[bB][uU][tT][tT][oO][nN] |
	[cC][aA][nN][vV][aA][sS] |
	[cC][aA][pP][tT][iI][oO][nN] |
	[cC][eE][nN][tT][eE][rR] |
	[cC][iI][tT][eE] |
	[cC][oO][dD][eE] |
	[cC][oO][lL] |
	[cC][oO][lL][gG][rR][oO][uU][pP] |
	[cC][oO][mM][mM][aA][nN][dD] |
	[cC][oO][mM][mM][eE][nN][tT] |
	[dD][dD] |
	[dD][aA][tT][aA][gG][rR][iI][dD] |
	[dD][aA][tT][aA][lL][iI][sS][tT] |
	[dD][aA][tT][aA][tT][eE][mM][pP][lL][aA][tT][eE] |
	[dD][eE][lL] |
	[dD][eE][tT][aA][iI][lL][sS] |
	[dD][fF][nN] |
	[dD][iI][aA][lL][oO][gG] |
	[dD][iI][rR] |
	[dD][iI][vV] |
	[dD][lL] |
	[dD][tT] |
	[eE][mM] |
	[eE][mM][bB][eE][dD] |
	[eE][vV][eE][nN][tT][sS][oO][uU][rR][cC][eE] |
	[fF][iI][eE][lL][dD][sS][eE][tT] |
	[fF][iI][gG][uU][rR][eE] |
	[fF][oO][nN][tT] |
	[fF][oO][oO][tT][eE][rR] |
	[fF][oO][rR][mM] |
	[fF][rR][aA][mM][eE] |
	[fF][rR][aA][mM][eE][sS][eE][tT] |
	[hH][123456] |
	[hH][eE][aA][dD] |
	[hH][eE][aA][dD][eE][rR] |
	[hH][rR] |
	[hH][tT][mM][lL] |
	[iI] |
	[iI][fF][rR][aA][mM][eE] |
	[iI][lL][aA][yY][eE][rR] |
	[iI][mM][gG] |
	[iI][nN][pP][uU][tT] |
	[iI][nN][sS] |
	[iI][sS][iI][nN][dD][eE][xX] |
	[kK][bB][dD] |
	[kK][eE][yY][gG][eE][nN] |
	[lL][aA][bB][eE][lL] |
	[lL][aA][yY][eE][rR] |
	[lL][eE][gG][eE][nN][dD] |
	[lL][iI] |
	[lL][iI][nN][kK] |
	[mM][aA][pP] |
	[mM][aA][rR][kK] |
	[mM][aA][rR][qQ][uU][eE][eE] |
	[mM][eE][nN][uU] |
	[mM][eE][tT][aA] |
	[mM][eE][tT][eE][rR] |
	[mM][uU][lL][tT][iI][cC][oO][lL] |
	[nN][aA][vV] |
	[nN][eE][sS][tT] |
	[nN][oO][bB][rR] |
	[nN][oO][eE][mM][bB][eE][dD] |
	[nN][oO][fF][rR][aA][mM][eE][sS] |
	[nN][oO][lL][aA][yY][eE][rR] |
	[nN][oO][sS][cC][rR][iI][pP][tT] |
	[oO][bB][jJ][eE][cC][tT] |
	[oO][lL] |
	[oO][pP][tT][gG][rR][oO][uU][pP] |
	[oO][pP][tT][iI][oO][nN] |
	[oO][uU][tT][pP][uU][tT] |
	[pP] |
	[pP][aA][rR][aA][mM] |
	[pP][lL][aA][iI][nN][tT][eE][xX][tT] |
	[pP][rR][eE] |
	[pP][rR][oO][gG][rR][eE][sS][sS] |
	[qQ] |
	[rR][uU][lL][eE] |
	[sS] |
	[sS][aA][mM][pP] |
	[sS][eE][cC][tT][iI][oO][nN] |
	[sS][eE][lL][eE][cC][tT] |
	[sS][eE][rR][vV][eE][rR] |
	[sS][mM][aA][lL][lL] |
	[sS][oO][uU][rR][cC][eE] |
	[sS][pP][aA][cC][eE][rR] |
	[sS][pP][aA][nN] |
	[sS][tT][rR][iI][kK][eE] |
	[sS][tT][rR][oO][nN][gG] |
	[sS][uU][bB] |
	[sS][uU][pP] |
	[tT][aA][bB][lL][eE] |
	[tT][bB][oO][dD][yY] |
	[tT][dD] |
	[tT][eE][xX][tT][aA][rR][eE][aA] |
	[tT][fF][oO][oO][tT] |
	[tT][hH] |
	[tT][hH][eE][aA][dD] |
	[tT][iI][mM][eE] |
	[tT][iI][tT][lL][eE] |
	[tT][rR] |
	[tT][tT] |
	[uU] |
	[uU][lL] |
	[vV][aA][rR] |
	[vV][iI][dD][eE][oO]    { addToken(Token.MARKUP_TAG_NAME); }
	{InTagIdentifier}		{ /* A non-recognized HTML tag name */ yypushback(yylength()); yybegin(INTAG); }
	.						{ /* Shouldn't happen */ yypushback(1); yybegin(INTAG); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG); return firstToken; }
}

<INTAG> {
	"/"						{ addToken(Token.MARKUP_TAG_DELIMITER); }
	{InTagIdentifier}			{ addToken(Token.MARKUP_TAG_ATTRIBUTE); }
	{Whitespace}				{ addToken(Token.WHITESPACE); }
	"="						{ addToken(Token.OPERATOR); }
	"/>"						{ yybegin(YYINITIAL); addToken(Token.MARKUP_TAG_DELIMITER); }
	">"						{ yybegin(YYINITIAL); addToken(Token.MARKUP_TAG_DELIMITER); }
	[\"]						{ start = zzMarkedPos-1; yybegin(INATTR_DOUBLE); }
	[\']						{ start = zzMarkedPos-1; yybegin(INATTR_SINGLE); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG); return firstToken; }
}

<INATTR_DOUBLE> {
	[^\"]*						{}
	[\"]						{ yybegin(INTAG); addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_DOUBLE); return firstToken; }
}

<INATTR_SINGLE> {
	[^\']*						{}
	[\']						{ yybegin(INTAG); addToken(start,zzStartRead, Token.MARKUP_TAG_ATTRIBUTE_VALUE); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.MARKUP_TAG_ATTRIBUTE_VALUE); addEndToken(INTERNAL_ATTR_SINGLE); return firstToken; }
}

<BOLDITALIC1> {
    [^*]*                       {}
    [*][*][*]                   { addToken(start,zzStartRead + 2, Token.FUNCTION); yybegin(YYINITIAL); }
    [*]                         {}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.FUNCTION); addEndToken(INTERNAL_IN_BOLDITALIC1); return firstToken; }
}
<BOLDITALIC2> {
    [^_]*                       {}
    [_][_][_]                   { addToken(start,zzStartRead + 2, Token.FUNCTION); yybegin(YYINITIAL); }
    [_]                         {}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.FUNCTION); addEndToken(INTERNAL_IN_BOLDITALIC2); return firstToken; }
}

<BOLD1> {
    [^*]*                       {}
    [*][*]                      { addToken(start,zzStartRead + 1, Token.RESERVED_WORD_2); yybegin(YYINITIAL); }
    [*]                         {}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.RESERVED_WORD_2); addEndToken(INTERNAL_IN_BOLD1); return firstToken; }
}
<BOLD2> {
    [^_]*                       {}
    [_][_]                      { addToken(start,zzStartRead + 1, Token.RESERVED_WORD_2); yybegin(YYINITIAL); }
    [_]                         {}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.RESERVED_WORD_2); addEndToken(INTERNAL_IN_BOLD2); return firstToken; }
}

<ITALIC1> {
    [^*]*                       {}
    [*]                         { addToken(start,zzStartRead, Token.DATA_TYPE); yybegin(YYINITIAL); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.DATA_TYPE); addEndToken(INTERNAL_IN_ITALIC1); return firstToken; }
}
<ITALIC2> {
    [^_]*                       {}
    [_]                         { addToken(start,zzStartRead, Token.DATA_TYPE); yybegin(YYINITIAL); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.DATA_TYPE); addEndToken(INTERNAL_IN_ITALIC2); return firstToken; }
}

<STRIKETHROUGH> {
    [^~]*                       {}
    [~][~]                      { addToken(start,zzStartRead + 1, Token.OPERATOR); yybegin(YYINITIAL); }
    [~]                         {}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.OPERATOR); addEndToken(INTERNAL_IN_STRIKETHROUGH); return firstToken; }
}

<CODE> {
    [^`]*                       {}
    [`]                         { addToken(start,zzStartRead, Token.PREPROCESSOR); yybegin(YYINITIAL); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.PREPROCESSOR); addEndToken(INTERNAL_IN_CODE); return firstToken; }
}

<SYNTAX_HIGHLIGHTING> {
    [^`]*                       {}
    [`][`][`]                   { addToken(start,zzStartRead + 2, Token.PREPROCESSOR); yybegin(YYINITIAL); }
    [`]                         {}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.PREPROCESSOR); addEndToken(INTERNAL_IN_SYNTAX_HIGHLIGHTING); return firstToken; }
}
