/*
 * Decompiled with CFR 0.152.
 */
package com.fizzed.stork.launcher;

import com.fizzed.stork.core.ArgumentException;
import com.fizzed.stork.launcher.Configuration;
import com.fizzed.stork.launcher.LauncherModel;
import freemarker.template.DefaultObjectWrapper;
import freemarker.template.ObjectWrapper;
import freemarker.template.Template;
import freemarker.template.TemplateExceptionHandler;
import freemarker.template.Version;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.TreeSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Generator {
    private static final Logger logger = LoggerFactory.getLogger(Generator.class);
    private static freemarker.template.Configuration fmconfig;

    private Path canonicalPath(File f) {
        return f.toPath().normalize();
    }

    public int generate(Configuration config, File outputDir) throws ArgumentException, IOException {
        return this.generate(Arrays.asList(config), outputDir);
    }

    public int generate(List<Configuration> configs, File outputDir) throws ArgumentException, IOException {
        if (outputDir == null) {
            throw new ArgumentException("No output dir was specified");
        }
        if (!outputDir.exists()) {
            if (!outputDir.mkdirs()) {
                throw new ArgumentException("Unable to crate output directory [" + outputDir + "]");
            }
            logger.info("Created dir {}", (Object)this.canonicalPath(outputDir));
        }
        if (!outputDir.isDirectory()) {
            throw new ArgumentException("Output directory [" + outputDir + "] exists but is not a directory");
        }
        if (!outputDir.canWrite()) {
            throw new ArgumentException("Output directory [" + outputDir + "] is not writable");
        }
        if (configs.isEmpty()) {
            logger.warn("No input configuration files (no stork launchers will be generated)");
            return 0;
        }
        int generated = 0;
        for (Configuration config : configs) {
            try {
                logger.info("Launcher {}", (Object)this.canonicalPath(config.getFile()));
                this.doGenerate(config, outputDir);
                ++generated;
            }
            catch (Exception e) {
                throw new IOException("Unable to cleanly generate launcher for [" + config.getFile() + "]", e);
            }
        }
        return generated;
    }

    private static freemarker.template.Configuration getOrCreateFreemarker() throws Exception {
        if (fmconfig != null) {
            return fmconfig;
        }
        freemarker.template.Configuration cfg = new freemarker.template.Configuration();
        cfg.setClassForTemplateLoading(Generator.class, "");
        cfg.setObjectWrapper((ObjectWrapper)new DefaultObjectWrapper());
        cfg.setDefaultEncoding("UTF-8");
        cfg.setTemplateExceptionHandler(TemplateExceptionHandler.RETHROW_HANDLER);
        cfg.setIncompatibleImprovements(new Version(2, 3, 20));
        fmconfig = cfg;
        return fmconfig;
    }

    private void doGenerate(Configuration config, File outputDir) throws ArgumentException, Exception {
        Configuration.Platform unixLauncherGeneratedVia = null;
        File binDir = Paths.get(outputDir.getPath(), config.getBinDir()).toFile();
        File shareDir = Paths.get(outputDir.getPath(), config.getShareDir()).toFile();
        TreeSet<Configuration.Platform> sortedPlatforms = new TreeSet<Configuration.Platform>(config.getPlatforms());
        for (Configuration.Platform platform : sortedPlatforms) {
            File launcherFile;
            logger.info(" {}", (Object)platform);
            LauncherModel model = new LauncherModel(config);
            if (platform == Configuration.Platform.LINUX || platform == Configuration.Platform.MAC_OSX) {
                if (unixLauncherGeneratedVia != null) {
                    logger.info("  script same as {}", unixLauncherGeneratedVia);
                } else {
                    binDir.mkdirs();
                    launcherFile = new File(binDir, config.getName());
                    if (config.getType() == Configuration.Type.CONSOLE) {
                        this.generateUnixConsoleLauncher(config, launcherFile, model);
                    } else if (config.getType() == Configuration.Type.DAEMON) {
                        this.generateUnixDaemonLauncher(config, launcherFile, model);
                    }
                    if (config.isIncludeJavaDetectHelper()) {
                        File helperDir = new File(shareDir, "helper");
                        helperDir.mkdirs();
                        File javaDetectFile = new File(helperDir, "java-detect");
                        this.generateUnixJavaDetectScript(javaDetectFile);
                    }
                    unixLauncherGeneratedVia = platform;
                }
                if (platform == Configuration.Platform.LINUX && config.getType() == Configuration.Type.DAEMON) {
                    File initdDir = new File(shareDir, "init.d");
                    initdDir.mkdirs();
                    File initdFile = new File(initdDir, config.getName() + ".init");
                    this.generateInitdScript(config, initdFile, model);
                    File systemdDir = new File(shareDir, "systemd");
                    systemdDir.mkdirs();
                    File systemdFile = new File(systemdDir, config.getName() + ".service");
                    this.generateSytemdScript(config, systemdFile, model);
                }
                if (platform != Configuration.Platform.MAC_OSX || config.getType() != Configuration.Type.DAEMON) continue;
                File osxDir = new File(shareDir, "osx");
                osxDir.mkdirs();
                File launchdFile = new File(osxDir, config.getDomain() + "." + config.getName() + ".plist");
                this.generateOSXLaunchdScript(config, launchdFile, model);
                continue;
            }
            if (platform == Configuration.Platform.WINDOWS) {
                if (config.getType() == Configuration.Type.CONSOLE) {
                    binDir.mkdirs();
                    launcherFile = new File(binDir, config.getName() + ".bat");
                    this.generateWindowsConsoleLauncher(config, launcherFile, model);
                    continue;
                }
                if (config.getType() != Configuration.Type.DAEMON) continue;
                Configuration.DaemonMethod dm = config.getPlatformDaemonMethod(Configuration.Platform.WINDOWS);
                switch (dm) {
                    case JSLWIN: {
                        this.generateWindowsJSLWinLauncher(config, binDir, model);
                        break;
                    }
                    case WINSW: {
                        this.generateWindowsWINSWLauncher(config, binDir, model);
                        break;
                    }
                    default: {
                        throw new ArgumentException("Unsupported daemon method [" + (Object)((Object)dm) + "] for platform WINDOWS");
                    }
                }
                continue;
            }
            throw new ArgumentException("Unsupported platform " + (Object)((Object)platform));
        }
    }

    private void generateUnixConsoleLauncher(Configuration config, File launcherFile, LauncherModel model) throws Exception {
        try (FileOutputStream fos = new FileOutputStream(launcherFile);
             OutputStreamWriter out = new OutputStreamWriter(fos);){
            this.processTemplate("linux/script-header.ftl", out, model);
            this.includeResource("linux/script-functions.sh", fos);
            this.processTemplate("linux/script-java.ftl", out, model);
            this.processTemplate("linux/script-console.ftl", out, model);
            launcherFile.setExecutable(true);
            logger.info("  script {}", (Object)launcherFile);
        }
    }

    private void generateUnixDaemonLauncher(Configuration config, File launcherFile, LauncherModel model) throws Exception {
        try (FileOutputStream fos = new FileOutputStream(launcherFile);
             OutputStreamWriter out = new OutputStreamWriter(fos);){
            this.processTemplate("linux/script-header.ftl", out, model);
            this.includeResource("linux/script-functions.sh", fos);
            this.processTemplate("linux/script-java.ftl", out, model);
            Configuration.DaemonMethod dm = config.getPlatformDaemonMethod(Configuration.Platform.LINUX);
            if (dm != Configuration.DaemonMethod.NOHUP) {
                throw new Exception("Unsupported daemon method [" + (Object)((Object)dm) + "] for platform LINUX");
            }
            this.processTemplate("linux/script-daemon-nohup.ftl", out, model);
            launcherFile.setExecutable(true);
            logger.info("  script {}", (Object)launcherFile);
        }
    }

    private void generateInitdScript(Configuration config, File initdFile, LauncherModel model) throws Exception {
        try (FileOutputStream fos = new FileOutputStream(initdFile);
             OutputStreamWriter out = new OutputStreamWriter(fos);){
            this.processTemplate("linux/initd-daemon.ftl", out, model);
            initdFile.setExecutable(true);
            logger.info("  init.d {}", (Object)initdFile);
        }
    }

    private void generateSytemdScript(Configuration config, File systemdFile, LauncherModel model) throws Exception {
        try (FileOutputStream fos = new FileOutputStream(systemdFile);
             OutputStreamWriter out = new OutputStreamWriter(fos);){
            this.processTemplate("linux/systemd-daemon.ftl", out, model);
            logger.info("  systemd {}", (Object)systemdFile);
        }
    }

    private void generateOSXLaunchdScript(Configuration config, File launchdFile, LauncherModel model) throws Exception {
        try (FileOutputStream fos = new FileOutputStream(launchdFile);
             OutputStreamWriter out = new OutputStreamWriter(fos);){
            this.processTemplate("osx/launchd.ftl", out, model);
            logger.info("  launchd {}", (Object)launchdFile);
        }
    }

    private void generateUnixJavaDetectScript(File file) throws Exception {
        try (FileOutputStream fos = new FileOutputStream(file);
             OutputStreamWriter out = new OutputStreamWriter(fos);){
            this.includeResource("linux/script-java-detect-header.sh", fos);
            this.includeResource("linux/script-functions.sh", fos);
            this.includeResource("linux/script-java-detect.sh", fos);
            file.setExecutable(true);
            logger.info("  helper {}", (Object)file);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void generateWindowsConsoleLauncher(Configuration config, File launcherFile, LauncherModel model) throws Exception {
        FileOutputStream fos = new FileOutputStream(launcherFile);
        OutputStreamWriter out = new OutputStreamWriter(fos);
        try {
            this.processTemplate("windows/batch-header.ftl", out, model);
            this.includeResource("windows/batch-find-java.bat", fos);
            this.processTemplate("windows/batch-java.ftl", out, model);
            this.processTemplate("windows/batch-console.ftl", out, model);
            this.processTemplate("windows/batch-footer.ftl", out, model);
            launcherFile.setExecutable(true);
            logger.info("  script {}", (Object)launcherFile);
        }
        finally {
            if (out != null) {
                ((Writer)out).close();
            }
            if (fos != null) {
                fos.close();
            }
        }
    }

    private void generateWindowsWINSWLauncher(Configuration config, File binDir, LauncherModel model) throws Exception {
        binDir.mkdirs();
        File serviceFile = new File(binDir, config.getName() + ".exe");
        File configFile = new File(binDir, config.getName() + ".xml");
        File netFile = new File(binDir, config.getName() + ".exe.config");
        this.copyResource("windows/winsw/winsw-1.16-bin.exe", serviceFile);
        logger.info("  created {}", (Object)serviceFile);
        this.copyResource("windows/winsw/winsw.exe.config", netFile);
        logger.info("  created {}", (Object)serviceFile);
        this.generateWindowsWINSWConfig(config, configFile, model);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void generateWindowsWINSWConfig(Configuration config, File configFile, LauncherModel model) throws Exception {
        FileOutputStream fos = new FileOutputStream(configFile);
        OutputStreamWriter out = new OutputStreamWriter(fos);
        try {
            this.processTemplate("windows/config-daemon-winsw.ftl", out, model);
        }
        finally {
            if (out != null) {
                ((Writer)out).close();
            }
            if (fos != null) {
                fos.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void generateWindowsJSLWinLauncher(Configuration config, File binDir, LauncherModel model) throws Exception {
        binDir.mkdirs();
        File launcherFile = new File(binDir, config.getName() + ".bat");
        FileOutputStream fos = new FileOutputStream(launcherFile);
        OutputStreamWriter out = new OutputStreamWriter(fos);
        try {
            this.processTemplate("windows/batch-header.ftl", out, model);
            this.includeResource("windows/batch-find-java.bat", fos);
            this.processTemplate("windows/batch-java.ftl", out, model);
            this.processTemplate("windows/batch-daemon-jslwin.ftl", out, model);
            this.processTemplate("windows/batch-footer.ftl", out, model);
            launcherFile.setExecutable(true);
            logger.info("  script {}", (Object)launcherFile);
        }
        finally {
            if (out != null) {
                ((Writer)out).close();
            }
            if (fos != null) {
                fos.close();
            }
        }
        File serviceFile = new File(binDir, config.getName() + "32.exe");
        File iniFile = new File(binDir, config.getName() + "32.ini");
        File service64File = new File(binDir, config.getName() + "64.exe");
        File ini64File = new File(binDir, config.getName() + "64.ini");
        this.copyResource("windows/jslwin/jsl.exe", serviceFile);
        logger.info("  helper {}", (Object)serviceFile);
        this.generateWindowsJSLWinINI(config, iniFile, model);
        logger.info("  helper {}", (Object)iniFile);
        this.copyResource("windows/jslwin/jsl64.exe", service64File);
        logger.info("  helper {}", (Object)service64File);
        this.generateWindowsJSLWinINI(config, ini64File, model);
        logger.info("  helper {}", (Object)ini64File);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void generateWindowsJSLWinINI(Configuration config, File iniFile, LauncherModel model) throws Exception {
        FileOutputStream fos = new FileOutputStream(iniFile);
        OutputStreamWriter out = new OutputStreamWriter(fos);
        try {
            this.processTemplate("windows/config-daemon-jslwin.ftl", out, model);
        }
        finally {
            if (out != null) {
                ((Writer)out).close();
            }
            if (fos != null) {
                fos.close();
            }
        }
    }

    private void processTemplate(String templateName, Writer out, Object model) throws Exception {
        freemarker.template.Configuration freemarker = Generator.getOrCreateFreemarker();
        Template template = freemarker.getTemplate(templateName);
        template.process(model, out);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyResource(String resourceName, File targetFile) throws Exception {
        try (FileOutputStream fos = new FileOutputStream(targetFile);){
            this.includeResource(resourceName, fos);
        }
    }

    private void includeResource(String resourceName, OutputStream os) throws Exception {
        int n;
        InputStream is = Generator.class.getResourceAsStream(resourceName);
        if (is == null) {
            throw new Exception("Unable to find resource " + resourceName);
        }
        byte[] buffer = new byte[1024];
        while ((n = is.read(buffer)) > -1) {
            os.write(buffer, 0, n);
        }
        is.close();
    }
}

