/*
 * Copyright 2017 FOLIO Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.folio_sec.reladomo.scala_api

import com.gs.fw.common.mithra.finder.Operation
import com.gs.fw.common.mithra.util.DoWhileProcedure
import com.gs.fw.finder.{ Navigation, OrderBy }

import scala.collection.JavaConverters._

/**
  * Represents a Scala facade of Reladomo's MithraTransactionalList.
  * The type must provide a MithraTransactionalList which is consistent with the Scala object's state.
  */
trait BiTemporalTransactionalList[TxObject <: BiTemporalTransactionalObject,
                                  MithraTxObject <: com.gs.fw.common.mithra.MithraDatedTransactionalObject]
    extends BiTemporalTransactionalListBase[TxObject, MithraTxObject] { self =>

  /**
    * Returns consistent MithraTransactionalList
    */
  def underlying: com.gs.fw.common.mithra.MithraDatedTransactionalList[MithraTxObject]

  def toScalaObject(mithraTxObject: MithraTxObject): TxObject

  // DomainList[MithraTxObject]

  // NOTE: The reason we gave up to have the generic type here is
  // the List class code generated by MithraGenerator doesn't have the type...
  // (as of version 16.3.0)
  def orderBy(orderBy: OrderBy[_]): self.type = {
    underlying.setOrderBy(orderBy.asInstanceOf[OrderBy[MithraTxObject]])
    this
  }

  def limit(size: Int): self.type = {
    underlying.setMaxObjectsToRetrieve(size)
    this
  }

  override def length: Int = count()

  override def apply(idx: Int): TxObject = toScalaObject(underlying.get(idx))

  override def iterator: Iterator[TxObject] = underlying.iterator().asScala.map(toScalaObject)

  /**
    * Iterates through the list using a com.gs.fw.common.mithra.list.cursor.Cursor.
    *
    * @param doWhileProcedure Executes this procedure so long as it returns true or there is no more work to do
    */
  def foreachWithCursor(doWhileProcedure: (TxObject) => Boolean) = {
    // Don't change the method to be SAM (due to overloaded methods, the compilation fails with it)
    underlying.forEachWithCursor(new DoWhileProcedure[MithraTxObject] {
      override def execute(obj: MithraTxObject): Boolean = doWhileProcedure.apply(toScalaObject(obj))
    })
  }

  /**
    * Iterates through the list using a com.gs.fw.common.mithra.list.cursor.Cursor.
    *
    * @param doWhileProcedure Executes this procedure so long as it returns true or there is no more work to do
    */
  def foreachWithCursor(doWhileProcedure: (TxObject) => Boolean, operation: Operation) = {
    // Don't change the method to be SAM (due to overloaded methods, the compilation fails with it)
    underlying.forEachWithCursor(new DoWhileProcedure[MithraTxObject] {
      override def execute(obj: MithraTxObject): Boolean = doWhileProcedure.apply(toScalaObject(obj))
    }, operation)
  }

  // TransactionalDomainList[MithraTxObject]

  /**
    * Performs #terminateAll with the current Transaction.
    *
    * The reasons to accept the implicit parameter are:
    * - avoiding to unintentionally issue insert/update/delete queries
    * - guaranteeing the existence of thread-local Reladomo transaction on the current thread.
    */
  def terminateAll()(implicit tx: Transaction): Unit = underlying.terminateAll()

  /**
    * Performs #purgeAll with the current Transaction.
    *
    * The reasons to accept the implicit parameter are:
    * - avoiding to unintentionally issue insert/update/delete queries
    * - guaranteeing the existence of thread-local Reladomo transaction on the current thread.
    */
  def purgeAll()(implicit tx: Transaction): Unit = underlying.purgeAll()

}
