/*
 * Copyright 2014 the original author or authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.foreach.common.spring.localization.text;

import com.foreach.common.spring.localization.Language;

import java.util.Collection;

/**
 * <p>LocalizedTextSet is a collection of LocalizedText instances scoped to a particular group.
 * It is a special structure to be used for:
 * <ul>
 * <li>looking up text in the collection</li>
 * <li>creating fields with default values if they do not yet exist</li>
 * <li>setting a field as autogenerated upon creation or used upon first call</li>
 * </ul>
 * </p>
 * <p>A LocalizedTextSet is hooked to a LocalizedTextService that should provide the actual implementation of
 * the LocalizedText backend.  Things like creating default values, saving in database, etc.</p>
 */
public interface LocalizedTextSet
{
	/**
	 * @return The application this set belongs to, new items in this set will be created in this application.
	 */
	String getApplication();

	/**
	 * @return The group this set represents, new items in this set will be created in this group.
	 */
	String getGroup();

	/**
	 * @return All items in this set.
	 */
	Collection<LocalizedText> getItems();

	/**
	 * Returns the value for a specific language of a text item.  If the text item is found and it is the first time
	 * it has been requested (based on the Used property of {@link LocalizedText}), this method will trigger a
	 * flagAsUsed call on the {@link LocalizedTextService} provided.
	 *
	 * @param label    Label of the text item.
	 * @param language Language for which we want the value.
	 * @return Value as a string.
	 */
	String getText( String label, Language language );

	/**
	 * Returns the value for a specific language of a text item.  If the text item is found and it is the first time
	 * it has been requested (based on the Used property of {@link LocalizedText}), this method will trigger a
	 * flagAsUsed call on the {@link LocalizedTextService} provided.
	 *
	 * @param label        Label of the text item.
	 * @param language     Language for which we want the value.
	 * @param defaultValue Value to return in case the text item does not yet exist.
	 * @return Value as a string.
	 */
	String getText( String label, Language language, String defaultValue );

    /**
     * Returns the value for a specific language of a text item.  If the text item is found and it is the first time
     * it has been requested (based on the Used property of {@link LocalizedText}), this method will trigger a
     * flagAsUsed call on the {@link LocalizedTextService} provided.
     *
     * @param label        Label of the text item.
     * @param language     Language for which we want the value.
     * @param defaultValue Value to return in case the text item does not yet exist.
     * @param storeDefault Whether the default value should be stored when not found
     * @return Value as a string.
     */
    String getText( String label, Language language, String defaultValue, boolean storeDefault );

    /**
	 * @param label Label of the item to search for.
	 * @return True if the item exists in this set, false if not.
	 */
	boolean exists( String label );

	/**
	 * @return Number of items (different label) in the set.
	 */
	int size();

	/**
	 * Reloads all items in this set from the backing data store.
	 */
	void reload();
}
