/*
 * Copyright 2014 the original author or authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.foreach.common.spring.localization.text;

import com.foreach.common.spring.localization.Language;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>LocalizedTextSet is a collection of LocalizedText instances scoped to a particular group.
 * It is a special structure to be used for:
 * <ul>
 * <li>looking up text in the collection</li>
 * <li>creating fields with default values if they do not yet exist</li>
 * <li>setting a field as autogenerated upon creation or used upon first call</li>
 * </ul>
 * </p>
 * <p>A LocalizedTextSet is hooked to a LocalizedTextService that should provide the actual implementation of
 * the LocalizedText backend.  Things like creating default values, saving in database, etc.</p>
 */
public class LocalizedTextSetImpl implements LocalizedTextSet {
    private final String application, group;
    private final LocalizedTextService localizedTextService;
    private Map<String, LocalizedText> textMap = new HashMap<String, LocalizedText>();

    /**
     * <p>Constructs a new LocalizedTextSet based on a collection of items.  The collection may contain duplicates
     * but in case of multiples the item with the highest index in the collection will be retained.</p>
     *
     * @param application          Name of the application that new items in this set should be created in.
     * @param group                Name of the group that new items in this set should be created in.
     * @param localizedTextService Service owning this set, where callbacks will occur based on set modifications.
     */
    protected LocalizedTextSetImpl( String application, String group, LocalizedTextService localizedTextService ) {
        this.application = application;
        this.group = group;
        this.localizedTextService = localizedTextService;

        reload();
    }

    /**
     * @return The application this set belongs to, new items in this set will be created in this application.
     */
    public final String getApplication() {
        return application;
    }

    /**
     * @return The group this set represents, new items in this set will be created in this group.
     */
    public final String getGroup() {
        return group;
    }

    /**
     * @return The LocalizedTextService this set is linked to.
     */
    final LocalizedTextService getLocalizedTextService() {
        return localizedTextService;
    }

    /**
     * @return All items in this set.
     */
    public final Collection<LocalizedText> getItems() {
        return textMap.values();
    }

    /**
     * Returns the value for a specific language of a text item.  If the text item is found and it is the first time
     * it has been requested (based on the Used property of {@link LocalizedText}), this method will trigger a
     * flagAsUsed call on the {@link LocalizedTextService} provided.
     *
     * @param label    Label of the text item.
     * @param language Language for which we want the value.
     * @return Value as a string.
     */
    public final String getText( String label, Language language ) {
        return getText( label, language, null );
    }

    /**
     * Returns the value for a specific language of a text item.  If the text item is found and it is the first time
     * it has been requested (based on the Used property of {@link LocalizedText}), this method will trigger a
     * flagAsUsed call on the {@link LocalizedTextService} provided.
     *
     * @param label        Label of the text item.
     * @param language     Language for which we want the value.
     * @param defaultValue Value to return in case the text item does not yet exist.
     * @return Value as a string.
     */
    public final String getText( String label, Language language, String defaultValue ) {
        return getText( label, language, defaultValue, true );
    }

    /**
     * Returns the value for a specific language of a text item.  If the text item is found and it is the first time
     * it has been requested (based on the Used property of {@link LocalizedText}), this method will trigger a
     * flagAsUsed call on the {@link LocalizedTextService} provided.
     *
     * @param label        Label of the text item.
     * @param language     Language for which we want the value.
     * @param defaultValue Value to return in case the text item does not yet exist.
     * @param storeDefault Whether the default value should be stored when not found.
     * @return Value as a string.
     */
    public String getText( String label, Language language, String defaultValue, boolean storeDefault ) {
        LocalizedText text = textMap.get( label );

        // If the requested item doesn't exist, assume it should and create it
        if( text == null && storeDefault ) {
            text = localizedTextService.saveDefaultText( application, group, label, defaultValue );
            textMap.put( text.getLabel(), text );
        }

        String result = null;

        // Upon first use, flag the text as being used
        if( text != null ) {
            if( !text.isUsed() ) {
                text.setUsed( true );
                localizedTextService.flagAsUsed( text );
            }
            result = text.getFieldsForLanguage( language ).getText();
        }

        return result;
    }

    /**
     * @param label Label of the item to search for.
     * @return True if the item exists in this set, false if not.
     */
    public final boolean exists( String label ) {
        return textMap.containsKey( label );
    }

    /**
     * @return Number of items (different label) in the set.
     */
    public final int size() {
        return textMap.size();
    }

    /**
     * Reloads all items in this set from the backing data store.
     */

    public final void reload() {
        HashMap<String, LocalizedText> updatedMap = new HashMap<String, LocalizedText>();
        List<LocalizedText> items = localizedTextService.getLocalizedTextItems( application, group );

        for( LocalizedText text : items ) {
            updatedMap.put( text.getLabel(), text );
        }

        textMap = updatedMap;
    }
}
