/*
 * Copyright (c) 2020. Fyber N.V -  All Rights Reserved
 */

package com.fyber.inneractive.sdk.external;

import android.os.Looper;
import android.view.View;
import android.view.ViewGroup;

import com.fyber.inneractive.sdk.activities.InneractiveFullscreenAdActivity.FullScreenRendererProvider;
import com.fyber.inneractive.sdk.logger.IACIlogDefines;
import com.fyber.inneractive.sdk.config.IADisplayConfigProvider;
import com.fyber.inneractive.sdk.config.IAUnitConfigProvider;
import com.fyber.inneractive.sdk.config.IAVideoConfigProvider;
import com.fyber.inneractive.sdk.config.enums.UnitDisplayType;
import com.fyber.inneractive.sdk.factories.IAAdViewRendererFactory;
import com.fyber.inneractive.sdk.factories.IAFullscreenAdRendererFactory;
import com.fyber.inneractive.sdk.flow.InneractiveAdSpotImpl;
import com.fyber.inneractive.sdk.flow.InneractiveRefreshableAdSpot;
import com.fyber.inneractive.sdk.flow.InneractiveUnitControllerImpl;
import com.fyber.inneractive.sdk.interfaces.InneractiveAdViewRenderer;
import com.fyber.inneractive.sdk.interfaces.InneractiveFullscreenAdRenderer;
import com.fyber.inneractive.sdk.util.IAConcurrencyUtil;
import com.fyber.inneractive.sdk.util.IAJavaUtil;
import com.fyber.inneractive.sdk.util.IAlog;

import java.util.HashSet;
import java.util.Set;

import static com.fyber.inneractive.sdk.util.IAlog.CI_LOG_LEVEL;

/**
 * A class for displaying ad view unit types. Both HTML and Video
 */
public class InneractiveAdViewUnitController extends InneractiveUnitControllerImpl<InneractiveAdViewEventsListener>
                                             implements InneractiveAdSpotImpl.RefreshAdListener, FullScreenRendererProvider {

    public static final int DISABLED_REFRESH_INTERVAL = -1;

    //commented out, might be useful we do support unit hotswap
    //private ViewGroup mViewGroupAdParentContainer;

    /** Keep a set of native renderers, for a complete cleanup */
    Set<InneractiveAdRenderer> mRenderers;
    /** Are we in the middle of destroy? If we do, do not send collapse */
    boolean mDestroying = false;

    private View mBoundView;

    protected int mAdContentWidth = -1;
    protected int mAdContentHeight = -1;

    /** Refresh units are seconds */
    private final static int MIN_REFRESH_INTERVAL = 30;
    private int mOverridenRefreshInterval = 0;

    public InneractiveAdViewUnitController() {
        mRenderers = new HashSet<InneractiveAdRenderer>();
    }

    /** TODO: Find a way that this interface is not exposed for publishers, and only for Heyzap */
    public InneractiveAdViewUnitController(int overridenRefreshInterval) {
        this();

        //if override == DISABLED_REFRESH_INTERVAL which is -1. we'll end up with the minimum.
        if (overridenRefreshInterval >= MIN_REFRESH_INTERVAL) {
            IAlog.d("InneractiveAdViewUnitController: Overriding remote config refresh interval to: %d", overridenRefreshInterval);
            mOverridenRefreshInterval = overridenRefreshInterval;
        } else {
            IAlog.d("InneractiveAdViewUnitController: Overriding remote config refresh interval - value too low. Setting to default: %d -> %d", overridenRefreshInterval, MIN_REFRESH_INTERVAL);
            mOverridenRefreshInterval = MIN_REFRESH_INTERVAL;
        }
        //Uncomment for debugging
        //mOverridenRefreshInterval = 6;
    }

    /** TODO: Find a way that this interface is not exposed for publishers, and only for Heyzap */
    public InneractiveAdViewUnitController(boolean disableRefresh) {
        this();
        if (disableRefresh) {
            mOverridenRefreshInterval = DISABLED_REFRESH_INTERVAL;
        }
    }

    @Override
    public boolean supports(InneractiveAdSpot adSpot) {
        IAUnitConfigProvider unitConfig = adSpot.getAdContent().getUnitConfig();

        if (unitConfig.getNative() != null) {
            return false;
        }

        // Is there a supported display unit?
        IADisplayConfigProvider displayUnit = unitConfig.getDisplay();
        if (displayUnit != null) {
            if (UnitDisplayType.BANNER.equals(displayUnit.getUnitDisplayType()) || UnitDisplayType.MRECT.equals(displayUnit.getUnitDisplayType())) {
                return true;
            }

            // If the returned unit is an interstitial unit, check that we don't have a registered fullscreen unit controller first
            if(UnitDisplayType.INTERSTITIAL.equals(displayUnit.getUnitDisplayType())) {
                return !adSpot.getCurrentProcessedRequest().getAllowFullscreen();
            }
        }

        // Is there a supported video unit?
        IAVideoConfigProvider videoUnit = unitConfig.getVideo();
        if (videoUnit != null) {
            if (UnitDisplayType.LANDSCAPE.equals(videoUnit.getUnitDisplayType()) || UnitDisplayType.SQUARE.equals(videoUnit.getUnitDisplayType()) ||
                    UnitDisplayType.MRECT.equals(videoUnit.getUnitDisplayType())) {
                return true;
            }
        }

        return false;
    }

    @Override
    public boolean supportsRefresh() {
        return true;
    }

    public InneractiveFullscreenAdRenderer getFullscreenRenderer() {
        InneractiveAdSpot adSpot = IAJavaUtil.safeReference(mAdSpot);
        InneractiveFullscreenAdRenderer fullscreenAdRenderer = IAFullscreenAdRendererFactory.get().createRenderer(adSpot);
        mRenderers.add(fullscreenAdRenderer);

        return fullscreenAdRenderer;
    }

    @Override
    public InneractiveContentController getSelectedContentController() {
        return mSelectedContentController;
    }

    @Override
    public void destroy() {
        if (Looper.myLooper() != null && Looper.getMainLooper() == Looper.myLooper()) {
            internalDestroy();
        } else {
            IAConcurrencyUtil.getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    internalDestroy();
                }
            });
        }
    }
    
    private void internalDestroy() {
        mDestroying = true;
        Set<InneractiveAdRenderer> clonedRenderers = new HashSet(mRenderers);
        for (InneractiveAdRenderer renderer : clonedRenderers) {
            renderer.destroy();
        }
    
        mRenderers.clear();
        //mViewGroupAdParentContainer = null;
        mBoundView = null;
        super.destroy();
    }

    /**
     * Open a full screen ad activity, which will be started using the given activity's context
     * @param parent parent view group. The ad view will be added as a child
     */
    public void bindView(ViewGroup parent) {
        //mViewGroupAdParentContainer = parent;


        IAlog.d("%sPPPP bindView called with parent: %s", logPrefix(), parent);
        InneractiveAdSpot adSpot = getAdSpot();
        if (adSpot == null) {
            IAlog.w("InneractiveFullscreenUnitController was not attached to an ad spot");
            return;
        }

        IAlog.d("%sPPPP bindView spot is %s", logPrefix(), adSpot);

//        if (mBoundView != null && mBoundView != parent) {
//            unbindView(mBoundView);
//        }

        // Is there already a renderer which is bounded to this view? Only update the view
        Set<InneractiveAdRenderer> clonedRenderers = new HashSet(mRenderers);
        for(InneractiveAdRenderer renderer : clonedRenderers) {
            if (renderer instanceof  InneractiveAdViewRenderer) {
                InneractiveAdViewRenderer adViewRenderer = (InneractiveAdViewRenderer)renderer;
                if (adViewRenderer.isBoundedToView(parent)) {
                    adViewRenderer.update();
                    IAlog.d("%sPPPP bindAdToRenderer returning an already attached renderer %s", logPrefix(), adViewRenderer);
                    // There is already a native renderer which is attached to this view. Return it
                    return;
                }
            }
        }

        mBoundView = parent;
        initializeRenderer(parent, adSpot);

    }

    private void initializeRenderer(ViewGroup parent, InneractiveAdSpot adSpot) {
        InneractiveAdViewRenderer renderer = IAAdViewRendererFactory.get().createRenderer(adSpot);
        if (mOverridenRefreshInterval != 0) {
            renderer.setOverridenRefreshInterval(mOverridenRefreshInterval);
        }

        if (renderer != null) {
            renderer.initialize(adSpot);
            selectContentController();
            renderer.renderAd(parent);
            mRenderers.add(renderer);
            IAlog.d("%sPPPP bindView created renderer %s", logPrefix(), renderer);
        } else {
            IAlog.w("%sCould not find a renderer for the given spot! Did you add the appropriate module to your project?", logPrefix());
        }
    }

    /**
     * Unbind a previously bounded view. Currently used by Mopub
     * @param view
     */
    public void unbindView(View view) {
        if (mBoundView != view) {
            IAlog.w("%s unbindView invoked with incorrect view, was - %s received - %s", logPrefix(), mBoundView, view);
        } else {
            mBoundView = null;
        }
        IAlog.d("%sPPPP unbindView called with %s", logPrefix(), view);
        IAlog.d("%sPPPP spot is %s", logPrefix(), mAdSpot);
        // Is there already a renderer which is bounded to this view?
        Set<InneractiveAdRenderer> clonedRenderers = new HashSet(mRenderers);
        for(InneractiveAdRenderer renderer : clonedRenderers) {
            if (renderer instanceof InneractiveAdViewRenderer) {
                InneractiveAdViewRenderer adViewRenderer = (InneractiveAdViewRenderer)renderer;
                if (adViewRenderer.isBoundedToView(view)) {
                    IAlog.d("%sPPPP unbindView unbinding renderer %s", logPrefix(), renderer);
                    adViewRenderer.unbindAd();
                    mRenderers.remove(adViewRenderer);
                    break;
                }
            }
        }
        //mViewGroupAdParentContainer = view == mViewGroupAdParentContainer ? null : mViewGroupAdParentContainer;
    }

    public void refreshAd() {
        IAlog.d("InneractiveFullscreenUnitController refreshAd called");
        InneractiveAdSpot spot = IAJavaUtil.safeReference(mAdSpot);
        if (spot != null) {
            if (spot instanceof InneractiveRefreshableAdSpot) {
                ((InneractiveRefreshableAdSpot) spot).refreshAd(this);
            }
        }
    }

    @Override
    public void onAdRefreshed(InneractiveAdSpot adSpot) {
        // For now, get the first ad view renderer and render the ad
        InneractiveAdViewRenderer target = null;
        Set<InneractiveAdRenderer> clonedRenderers = new HashSet(mRenderers);
        for (InneractiveAdRenderer renderer : clonedRenderers) {
            if (renderer instanceof InneractiveAdViewRenderer && ((InneractiveAdViewRenderer) renderer).canRenderAdContent(adSpot.getAdContent())) {
                target = (InneractiveAdViewRenderer) renderer;
                break;
            }
            //we don't support changing between ad types on the same unit (i.e. display being changed to video) yet
//            else if (renderer instanceof InneractiveAdViewRenderer && ((InneractiveAdViewRenderer) renderer).isBoundedToView(mViewGroupAdParentContainer)) {
//                ((InneractiveAdViewRenderer) renderer).unbindAd();
//            }
        }

        if (target != null) {
            target.renderAd(null);
        } else {
            //we don't support changing between ad types on the same unit (i.e. display being changed to video) yet
//            mRenderers.clear();
//            initializeRenderer(mViewGroupAdParentContainer, adSpot);
            onAdRefreshFailed(adSpot, InneractiveErrorCode.SDK_INTERNAL_ERROR);
        }
        try {
            if (adSpot instanceof InneractiveAdSpotImpl) {
                InneractiveAdSpotImpl adSpotImp = (InneractiveAdSpotImpl) adSpot;
                IAlog.log(CI_LOG_LEVEL, null, "%s %s", IACIlogDefines.AD_REFRESH, adSpotImp.getAdSource().getAdFetcher().getUrl());
            }
        } catch (Exception ignored) {}
        IAlog.d("InneractiveFullscreenUnitController onAdRefreshed called");
    }

    @Override
    public void onAdRefreshFailed(InneractiveAdSpot adSpot, InneractiveErrorCode errorCode) {
        // For now, get the first ad view renderer and render the ad
        Set<InneractiveAdRenderer> clonedRenderers = new HashSet(mRenderers);
        for (InneractiveAdRenderer renderer : clonedRenderers) {
            if (renderer instanceof InneractiveAdViewRenderer) {
                ((InneractiveAdViewRenderer)renderer).onAdRefreshFailed();
                break;
            }
        }
    }

    /** For internal use only */
    public void unbindFullscreenRenderer(InneractiveFullscreenAdRenderer fullscreenAdRenderer) {
        IAlog.d("%sremoving full screen ad renderer %s", logPrefix(), fullscreenAdRenderer);
        if (mRenderers != null) {
            mRenderers.remove(fullscreenAdRenderer);
        }

        if (!mDestroying) {
            // Notify all the native ad renderers, that we are no longer in fullscreen
            Set<InneractiveAdRenderer> clonedRenderers = new HashSet(mRenderers);
            for (InneractiveAdRenderer renderer : clonedRenderers) {
                if (renderer instanceof InneractiveAdViewRenderer) {
                    ((InneractiveAdViewRenderer) renderer).fullscreenExited();

                    // Report collapsed to external listener
                    if (mEventsListener != null && mAdSpot != null && mAdSpot.get() != null) {
                        mEventsListener.onAdCollapsed(mAdSpot.get());
                    }
                }
            }
        }
    }

    @Override
    public boolean canRefreshAd() {
        if (supportsRefresh()) {
            for (InneractiveAdRenderer adRenderer : mRenderers) {
                if (adRenderer.canRefreshAd() == false) {
                    return false;
                }
            }

            return true;
        }

        return false;
    }

    public int getAdContentWidth() {
        for (InneractiveAdRenderer renderer : mRenderers) {
            if (renderer instanceof InneractiveAdViewRenderer) {
                int adContentWidth = ((InneractiveAdViewRenderer) renderer).getAdContentWidth();
                if (adContentWidth > 0) {
                    return adContentWidth;
                }
            }
        }
        return mAdContentWidth;
    }

    public int getAdContentHeight() {
        for (InneractiveAdRenderer renderer : mRenderers) {
            if (renderer instanceof InneractiveAdViewRenderer) {
                int adContentHeight = ((InneractiveAdViewRenderer) renderer).getAdContentHeight();
                if (adContentHeight > 0) {
                    return adContentHeight;
                }
            }
        }
        return mAdContentHeight;
    }
}
