/*
 * Copyright (c) 2020. Fyber N.V -  All Rights Reserved
 */

package com.fyber.inneractive.sdk.external;

import android.annotation.TargetApi;

/** Base interface for a unit controller. Concrete classes implements specific unit types */
public interface InneractiveUnitController<EL extends InneractiveUnitController.EventsListener> {

    /**
     * will send event 998 CAUGHT_EXCEPTION
     */
    public static class AdDisplayError extends Exception {

        /**
         * Constructs a new exception with {@code null} as its detail message.
         * The cause is not initialized, and may subsequently be initialized by a
         * call to {@link #initCause}.
         */
        public AdDisplayError() {
            super();
        }

        /**
         * Constructs a new exception with the specified detail message.  The
         * cause is not initialized, and may subsequently be initialized by
         * a call to {@link #initCause}.
         *
         * @param message the detail message. The detail message is saved for
         * later retrieval by the {@link #getMessage()} method.
         */
        public AdDisplayError(String message) {
            super(message);
        }

        /**
         * Constructs a new exception with the specified detail message and
         * cause.  <p>Note that the detail message associated with
         * {@code cause} is <i>not</i> automatically incorporated in
         * this exception's detail message.
         *
         * @param message the detail message (which is saved for later retrieval
         * by the {@link #getMessage()} method).
         * @param cause the cause (which is saved for later retrieval by the
         * {@link #getCause()} method).  (A <tt>null</tt> value is
         * permitted, and indicates that the cause is nonexistent or
         * unknown.)
         * @since 1.4
         */
        public AdDisplayError(String message, Throwable cause) {
            super(message, cause);
        }

        /**
         * Constructs a new exception with the specified cause and a detail
         * message of <tt>(cause==null ? null : cause.toString())</tt> (which
         * typically contains the class and detail message of <tt>cause</tt>).
         * This constructor is useful for exceptions that are little more than
         * wrappers for other throwables
         *
         * @param cause the cause (which is saved for later retrieval by the
         * {@link #getCause()} method).  (A <tt>null</tt> value is
         * permitted, and indicates that the cause is nonexistent or
         * unknown.)
         * @since 1.4
         */
        public AdDisplayError(Throwable cause) {
            super(cause);
        }

        /**
         * Constructs a new exception with the specified detail message,
         * cause, suppression enabled or disabled, and writable stack
         * trace enabled or disabled.
         *
         * @param message the detail message.
         * @param cause the cause.  (A {@code null} value is permitted,
         * and indicates that the cause is nonexistent or unknown.)
         * @param enableSuppression whether or not suppression is enabled
         * or disabled
         * @param writableStackTrace whether or not the stack trace should
         * be writable
         * @since 1.7
         */
        @TargetApi(24)
        public AdDisplayError(String message, Throwable cause, boolean enableSuppression,
            boolean writableStackTrace) {
            super(message, cause, enableSuppression, writableStackTrace);
        }
    }

    /**
     * External interface. Can be implemented by the Publisher, in order to get back SDK events
     */
    public interface EventsListener {
        /**
         * Called after Inneractive tracked an ad impression
         * @param adSpot
         */
        public void onAdImpression(InneractiveAdSpot adSpot);

        /**
         * Called after Inneractive tracked an ad click
         * @param adSpot
         */
        public void onAdClicked(InneractiveAdSpot adSpot);

        /**
         * Called after Inneractive decided to open another application, due to a click
         * @param adSpot
         */
        public void onAdWillCloseInternalBrowser(InneractiveAdSpot adSpot);

        /**
         * Called after Inneractive decided to open another application, due to a click
         * @param adSpot
         */
        public void onAdWillOpenExternalApp(InneractiveAdSpot adSpot);

        /**
         * Called when an ad has entered an error state, this will only happen when the ad is being shown
         * @param adSpot the relevant ad spot
         */
        public void onAdEnteredErrorState(InneractiveAdSpot adSpot, AdDisplayError error);
    }

    //javadoc adapted from AnimatorListenerAdapter
    /**
     * This adapter class provides empty implementations of the methods from {@link EventsListener}.
     * Any custom listener that cares only about a subset of the methods of this listener can
     * simply subclass this adapter class instead of implementing the interface directly.
     */
    public abstract class EventsListenerAdapter implements EventsListener {

        /**
         * Called after Inneractive tracked an ad impression
         */
        @Override
        public void onAdImpression(InneractiveAdSpot adSpot) {

        }

        /**
         * Called after Inneractive tracked an ad click
         */
        @Override
        public void onAdClicked(InneractiveAdSpot adSpot) {

        }


        /**
         * Called after Inneractive decided to open another application, due to a click
         */
        @Override
        public void onAdWillCloseInternalBrowser(InneractiveAdSpot adSpot) {

        }

        /**
         * Called after Inneractive decided to open another application, due to a click
         */
        @Override
        public void onAdWillOpenExternalApp(InneractiveAdSpot adSpot) {

        }
    }


    /**
     * Setter for the specific events listener
     * @param eventsListener
     */
    public void setEventsListener(EL eventsListener);

    /** Getter for events listener */
    public EL getEventsListener();

    /** Call in order to get the Spot, which is using this unit controller */
    public InneractiveAdSpot getAdSpot();

    /**
     * Add a specific content controller to this unit controller
     * @param contentController
     */
    public void addContentController(InneractiveContentController contentController);

    /** Returns the content controller, which was selected for the current ad */
    public InneractiveContentController getSelectedContentController();

    /** Call in order to destroy the unit content */
    public void destroy();
}
