/*
 * Copyright (c) 2020. Fyber N.V -  All Rights Reserved
 */

package com.fyber.inneractive.sdk.external;

import com.fyber.inneractive.sdk.annotations.Nullable;
import com.fyber.inneractive.sdk.config.IAUnitConfigProvider;
import com.fyber.inneractive.sdk.util.IAlog;

import java.util.concurrent.TimeUnit;

/**
 * Base class for defining an ad request
 * <br>Currently used only for native ads. Banners and Interstitials still use a direct API
 */
public class InneractiveAdRequest extends com.fyber.inneractive.sdk.flow.InternalAdRequest {

	public static final double FLOOR_PRICE_MAX_VALUE = 400_000;
	public static final double FLOOR_PRICE_MIN_VALUE = 0;

	// Members

	// Global request params
	/** Spot id */
	private final String mSpotId;
	/** The selected unit config, which will be used when reusing this request */
	private IAUnitConfigProvider selectedUnitConfig;
	/** The custom floor price */
	@Nullable
	private Double mFloorPrice;

	/**
	 * Constructor
	 * @param spotId AKA appId. As provided in the Inneractive console
	 */
	public InneractiveAdRequest(String spotId) {
		mSpotId = spotId;
	}

	/** Returns the attached spot id */
	public String getSpotId() {
		return mSpotId;
	}
	
	/**
	 * @deprecated use {@link InneractiveAdManager#setUserParams(InneractiveUserConfig)})}
	 * Set user related parameters specific for this ad
	 * @param userParams InneractiveUserConfig object, defining user related parameters
	 */
	@Deprecated
	public void setUserParams(InneractiveUserConfig userParams) {
		InneractiveAdManager.setUserParams(userParams);
	}

	/**
	 * @deprecated use {@link com.fyber.inneractive.sdk.external.InneractiveAdManager#setMuteVideo(boolean)}
	 * Set muteVideo for full screen videos
	 * @param muteVideo set to true or false
	 */
	@Deprecated
	public void setMuteVideo(boolean muteVideo) {
		InneractiveAdManager.setMuteVideo(muteVideo);
	}

	// Getters
	/** Getter for user parameters */
	@Deprecated
	public InneractiveUserConfig getUserParams() {
		return InneractiveAdManager.getUserParams();
	}

	/** Call this if you want this request to use a specific unit configuration */
	public void setSelectedUnitConfig(IAUnitConfigProvider unitConfig) {
		selectedUnitConfig = unitConfig;
	}

	/** Getter for selected unit config */
	public IAUnitConfigProvider getSelectedUnitConfig() {
		return selectedUnitConfig;
	}

	/** Getter for mute video state */
	@Deprecated
	public boolean getMuteVideo() {
		return InneractiveAdManager.getMuteVideo();
    }

	/**
	 * Sets a custom floor price in USD.
	 * Values are automatically adjusted: negative or NaN → {@link InneractiveAdRequest#FLOOR_PRICE_MIN_VALUE},
	 * above {@link InneractiveAdRequest#FLOOR_PRICE_MAX_VALUE} → {@link InneractiveAdRequest#FLOOR_PRICE_MAX_VALUE}.
	 *
	 * @param price the requested floor price in USD
	 */
	public void setFloorPrice(double price) {
		double clamped = Double.isNaN(price) ? FLOOR_PRICE_MIN_VALUE :
						Math.max(FLOOR_PRICE_MIN_VALUE, Math.min(price, FLOOR_PRICE_MAX_VALUE));

		if (clamped != price) {
			IAlog.w("Invalid floor price: %.2f — adjusted to %.2f.", price, clamped);
		}

		mFloorPrice = clamped;
	}

	/**
	 * Returns the current floor price set.
	 * <p>
	 * The returned value is the adjusted floor price if the originally
	 * requested value was out of bounds (negative/NaN → {@link InneractiveAdRequest#FLOOR_PRICE_MIN_VALUE}, above {@link InneractiveAdRequest#FLOOR_PRICE_MAX_VALUE} → {@link InneractiveAdRequest#FLOOR_PRICE_MAX_VALUE}).
	 * May return {@code null} if no floor price has been set.
	 *
	 * @return the floor price in USD, or {@code null} if unset
	 */
	@Nullable
	public Double getFloorPrice() {
		return mFloorPrice;
	}

}
