/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.api;

import static com.geotab.api.WebMethods.Authenticate;
import static com.geotab.http.invoker.ServerInvoker.DEFAULT_TIMEOUT;
import static java.util.Collections.singleton;
import static java.util.Collections.unmodifiableList;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.CollectionType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.geotab.http.exception.InvalidUserException;
import com.geotab.http.invoker.ServerInvoker;
import com.geotab.http.request.AuthenticatedRequest;
import com.geotab.http.request.BaseRequest;
import com.geotab.http.request.MultiCallRequest;
import com.geotab.http.request.param.AuthenticateParameters;
import com.geotab.http.request.param.AuthenticatedParameters;
import com.geotab.http.request.param.MultiCallParameters;
import com.geotab.http.response.BaseResponse;
import com.geotab.model.FeedResult;
import com.geotab.model.Id;
import com.geotab.model.entity.Entity;
import com.geotab.model.login.Credentials;
import com.geotab.model.login.LoginResult;
import com.geotab.model.serialization.ApiJsonSerializer;
import com.geotab.util.Util;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import lombok.extern.slf4j.Slf4j;
import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.jetbrains.annotations.Nullable;

@Slf4j
public class GeotabApi implements Api {

  public static final String THIS_SERVER = "ThisServer";
  public static final String DEFAULT_SERVER = "my.geotab.com";
  public static final String PROTOCOL = "https://";

  protected final Credentials credentials;
  protected final String server;
  protected final int timeout;
  protected final AtomicReference<LoginResult> auth = new AtomicReference<>();
  protected final AtomicReference<ServerInvoker> invoker = new AtomicReference<>();

  /**
   * Create new api instance.
   *
   * @param credentials {@link Credentials} used to authenticate.
   */
  public GeotabApi(Credentials credentials) {
    this(credentials, DEFAULT_SERVER, DEFAULT_TIMEOUT);
  }

  /**
   * Create new api instance.
   *
   * @param credentials {@link Credentials} used to authenticate.
   * @param server Server url without protocol. Example: my.geotab.com
   * @param timeout Request timeout
   */
  public GeotabApi(Credentials credentials, String server, int timeout) {
    this(credentials, server, timeout, null, null);
  }

  /**
   * Create new api instance.
   *
   * @param credentials {@link Credentials} used to authenticate.
   * @param server Server url without protocol. Example: my.geotab.com
   * @param timeout Request timeout
   * @param servicePath Service path. Default value is: apiv1
   * @param httpClient Custom {@link HttpClient} in case custom configuration is needed
   */
  public GeotabApi(Credentials credentials, String server, int timeout, String servicePath,
      CloseableHttpClient httpClient) {
    if (credentials == null) throw new IllegalArgumentException("Credentials not provided");
    log.debug("API params: credentials = {},  server = {}, timeout = {}, servicePath = {}, {}",
        credentials, server, timeout, servicePath, httpClient != null ? "custom http client" : "default http client");

    credentials.validate();
    this.credentials = credentials;
    this.timeout = timeout;
    this.server = Optional.ofNullable(server).orElse(DEFAULT_SERVER);
    this.invoker.set(buildServerInvoker(PROTOCOL + this.server, this.timeout, servicePath, httpClient));
  }

  private ObjectMapper om() {
    return ApiJsonSerializer.getInstance().getObjectMapper();
  }

  private TypeFactory tf() {
    return om().getTypeFactory();
  }

  @Override
  public boolean isAuthenticated() {
    LoginResult loginResult = auth.get();
    return loginResult != null && loginResult.getCredentials() != null;
  }

  @Override
  public LoginResult authenticate() {
    if (isAuthenticated()) {
      return auth.get();
    }

    @Nullable LoginResult loginResult;
    if (Util.isNotEmpty(credentials.getSessionId())) {
      log.debug("Geotab session id is provided as part of the credentials; will not call Geotab Authenticate method");
      loginResult = LoginResult.builder().path(server)
          .credentials(Credentials.builder()
              .database(credentials.getDatabase())
              .userName(credentials.getUserName())
              .sessionId(credentials.getSessionId())
              .build())
          .build();
      auth.set(loginResult);
    } else {
      log.debug("Calling Geotab Authenticate method ...");
      AuthenticatedRequest<AuthenticateParameters> req = prepareMethod(Authenticate, AuthenticateParameters.builder()
          .database(credentials.getDatabase()).userName(credentials.getUserName()).password(credentials.getPassword())
          .build());
      loginResult = invoker.get().invoke(req, Authenticate).orElse(null);
      auth.set(loginResult);
      log.info("Geotab Authenticate is successful");
    }

    // Change invoker to use the request URL to that of the db as returned by Authenticate.
    if (!THIS_SERVER.equals(auth.get().getPath())) {
      invoker.get().setUrl(buildServerPath());
    }

    return loginResult;
  }

  @Override
  public <O extends BaseResponse<T>, T> Optional<T> call(AuthenticatedRequest<?> in, Class<O> outT) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (outT == null) throw new IllegalArgumentException("Response type must be provided");
    return doCall(in, () -> invoker.get().invoke(in, outT));
  }

  @Override
  public <T> Optional<T> callResult(AuthenticatedRequest<?> in, Class<T> outT) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (outT == null) throw new IllegalArgumentException("Response type must be provided");
    JavaType responseT = tf().constructParametricType(BaseResponse.class, outT);
    return doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  @Override
  public <P extends AuthenticatedParameters, R> Optional<R> callMethod(MethodDescriptor<P, R> method, P params) {
    AuthenticatedRequest<P> in = prepareMethod(method, params);
    JavaType responseT = tf().constructParametricType(BaseResponse.class, method.resultType());
    return doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  private <P extends AuthenticatedParameters, R> AuthenticatedRequest<P> prepareMethod(
      MethodDescriptor<P, R> method, P params) {
    if (method == null) throw new IllegalArgumentException("Method expected");
    if (params == null) throw new IllegalArgumentException("Parameters expected");
    return AuthenticatedRequest.<P>authRequestBuilder().method(method.name).params(params).build();
  }

  @Override
  public <T> Optional<List<T>> callResultList(AuthenticatedRequest<?> in, Class<T> outT) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (outT == null) throw new IllegalArgumentException("Response type must be provided");
    CollectionType resultT = tf().constructCollectionType(List.class, outT);
    JavaType responseT = tf().constructParametricType(BaseResponse.class, resultT);
    return doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  @Override
  public Optional<Id> callAdd(AuthenticatedRequest<?> in) {
    JavaType responseT = tf().constructParametricType(BaseResponse.class, prepareAdd(in));
    return doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  private JavaType prepareAdd(AuthenticatedRequest<?> in) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (!Api.Add.equals(in.getMethod())) throw new IllegalArgumentException("Request method 'Add' expected");
    return tf().constructType(Id.class);
  }

  @Override
  public <T extends Entity> Optional<List<T>> callGet(AuthenticatedRequest<?> in, Class<T> outT) {
    JavaType responseT = tf().constructParametricType(BaseResponse.class, prepareGet(in, outT));
    return doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  private <T extends Entity> JavaType prepareGet(AuthenticatedRequest<?> in, Class<T> outT) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (!Api.Get.equals(in.getMethod())) throw new IllegalArgumentException("Request method 'Get' expected");
    if (outT == null) throw new IllegalArgumentException("Response type must be provided");
    return tf().constructCollectionType(List.class, outT);
  }

  @Override
  public Optional<Integer> callGetCountOf(AuthenticatedRequest<?> in) {
    JavaType responseT = tf().constructParametricType(BaseResponse.class, prepareGetCountOf(in));
    return doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  private JavaType prepareGetCountOf(AuthenticatedRequest<?> in) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (!GetCountOf.equals(in.getMethod())) throw new IllegalArgumentException("Request method 'GetCountOf' expected");
    return tf().constructType(Integer.class);
  }

  @Override
  public <T extends Entity> Optional<FeedResult<T>> callGetFeed(AuthenticatedRequest<?> in, Class<T> outT) {
    JavaType responseT = tf().constructParametricType(BaseResponse.class, prepareGetFeed(in, outT));
    return doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  private <T extends Entity> JavaType prepareGetFeed(AuthenticatedRequest<?> in, Class<T> outT) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (!Api.GetFeed.equals(in.getMethod())) throw new IllegalArgumentException("Request method 'GetFeed' expected");
    if (outT == null) throw new IllegalArgumentException("Response type must be provided");
    return tf().constructParametricType(FeedResult.class, outT);
  }

  @Override
  public void callSet(AuthenticatedRequest<?> in) {
    JavaType responseT = tf().constructParametricType(BaseResponse.class, prepareSet(in));
    doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  private JavaType prepareSet(AuthenticatedRequest<?> in) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (!Api.Set.equals(in.getMethod())) throw new IllegalArgumentException("Request method 'Set' expected");
    return tf().constructType(Void.class);
  }

  @Override
  public void callRemove(AuthenticatedRequest<?> in) {
    JavaType responseT = tf().constructParametricType(BaseResponse.class, prepareRemove(in));
    doCall(in, () -> invoker.get().invokeUnsafe(in, responseT));
  }

  private JavaType prepareRemove(AuthenticatedRequest<?> in) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (!Api.Remove.equals(in.getMethod())) throw new IllegalArgumentException("Request method 'Remove' expected");
    return tf().constructType(Void.class);
  }

  private <T> Optional<T> doCall(AuthenticatedRequest<?> in, Supplier<Optional<T>> supplier) {
    if (in == null) throw new IllegalArgumentException("Request expected");
    if (in.getMethod() == null) throw new IllegalArgumentException("Request method must be provided");

    boolean retry = false;
    while (true) {

      // Must be authenticated; will auto authenticate if not authenticated.
      if ((!in.getCredentials().isPresent() || retry) && !isAuthenticated()) {
        authenticate();
      }

      if (!in.getCredentials().isPresent()) {
        LoginResult loginResult = auth.get();
        in.setCredentials(loginResult.getCredentials());
      }

      // Invoke
      try {
        return supplier.get();
      } catch (InvalidUserException ex) {
        in.setCredentials(null);
        auth.set(null);
        if (retry || Util.isEmpty(credentials.getPassword())) {
          // If we retried request after a successful re-authentication throw, there is something happening between the
          // time we authenticate and try to make a request Must have password to login again
          log.error("Geotab call failed due to authentication; trying to re-authenticate also failed.", ex);
          throw ex;
        }

        // Token may have expired or DB may have moved so try to authenticate and retry request
        log.warn("Geotab call failed due to authentication; trying to re-authenticate and re-call");
        retry = true;
      }
    }
  }

  @Override
  public <O extends BaseResponse<T>, T> Optional<T> multiCall(MultiCallRequest in, Class<O> outT) {
    return call(in, outT);
  }

  @Override
  public <T> Optional<List<T>> uniformMultiCall(List<? extends BaseRequest<?>> calls, Class<T> outT) {
    if (outT == null) throw new IllegalArgumentException("Response type must be provided");
    MultiCallRequest request = MultiCallRequest.multiCallRequestBuilder()
        .params(MultiCallParameters.multiCallParamsBuilder().calls(unmodifiableList(calls)).build())
        .build();
    return doCall(request, () -> invoker.get().invokeJson(request).map(multiCallResult -> {
      ObjectMapper om = om();
      try {
        if (!multiCallResult.isArray()) {
          throw new RuntimeException("Invalid response, cause: multiCall expect array result");
        }
        List<T> out = new ArrayList<>();
        for (JsonNode perCallResult : multiCallResult) {
          for (JsonNode typeResult : perCallResult.isArray() ? perCallResult : singleton(perCallResult)) {
            out.add(om.treeToValue(typeResult, outT));
          }
        }
        return out;
      } catch (JsonProcessingException e) {
        throw new RuntimeException("Invalid response, cause: decoding JSON error", e);
      }
    }));
  }

  @Override
  public MultiCallBuilder buildMultiCall() {
    return new MultiCallBuilder() {
      final List<MyCall<?>> calls = new ArrayList<>();
      Map<String, Object> httpHeaders;

      <T> MyCall<T> add(AuthenticatedRequest<?> in, JavaType outT) {
        MyCall<T> call = new MyCall<>(in, outT);
        calls.add(call);
        return call;
      }

      public MultiCallBuilder httpHeaders(Map<String, Object> httpHeaders) {
        this.httpHeaders = httpHeaders;
        return this;
      }

      public <O extends BaseResponse<T>, T> Supplier<T> call(AuthenticatedRequest<?> in, Class<O> outT) {
        JavaType asBaseResponse = tf().constructType(outT).findSuperType(BaseResponse.class);
        if (asBaseResponse == null) throw new IllegalArgumentException("subtype of BaseResponse expected");
        return add(in, asBaseResponse.containedType(0));
      }

      @Override
      public <T> Supplier<T> callResult(AuthenticatedRequest<?> in, Class<T> outT) {
        if (in == null) throw new IllegalArgumentException("Request expected");
        if (outT == null) throw new IllegalArgumentException("Response type must be provided");
        return add(in, tf().constructType(outT));
      }

      @Override
      public <T> Supplier<List<T>> callResultList(AuthenticatedRequest<?> in, Class<T> outT) {
        if (in == null) throw new IllegalArgumentException("Request expected");
        if (outT == null) throw new IllegalArgumentException("Response type must be provided");
        return add(in, tf().constructCollectionType(List.class, outT));
      }

      @Override
      public <P extends AuthenticatedParameters, R> Supplier<R> callMethod(MethodDescriptor<P, R> in, P params) {
        return add(prepareMethod(in, params), in.resultType());
      }

      @Override
      public Supplier<Id> callAdd(AuthenticatedRequest<?> in) {
        return add(in, prepareAdd(in));
      }

      @Override
      public <T extends Entity> Supplier<List<T>> callGet(AuthenticatedRequest<?> in, Class<T> outT) {
        return add(in, prepareGet(in, outT));
      }

      @Override
      public Supplier<Integer> callGetCountOf(AuthenticatedRequest<?> in) {
        return add(in, prepareGetCountOf(in));
      }

      @Override
      public <T extends Entity> Supplier<FeedResult<T>> callGetFeed(AuthenticatedRequest<?> in, Class<T> outT) {
        return add(in, prepareGetFeed(in, outT));
      }

      @Override
      public void callSet(AuthenticatedRequest<?> in) {
        add(in, prepareSet(in));
      }

      @Override
      public void callRemove(AuthenticatedRequest<?> in) {
        add(in, prepareRemove(in));
      }

      public void execute() {
        List<BaseRequest<?>> requests = new ArrayList<>(calls.size());
        for (MyCall<?> call : calls) {
          requests.add(call.in);
        }

        MultiCallRequest request = MultiCallRequest.multiCallRequestBuilder()
            .params(MultiCallParameters.multiCallParamsBuilder().calls(requests).build())
            .httpHeaders(httpHeaders)
            .build();

        GeotabApi.this.doCall(request, () -> invoker.get().invokeJson(request).map(multiCallResult -> {
          ObjectMapper om = om();
          try {
            if (!multiCallResult.isArray()) {
              throw new RuntimeException("Invalid response, cause: multiCall expect array result");
            }
            if (multiCallResult.size() != calls.size()) {
              throw new RuntimeException("Invalid response, cause: multiCall array size mismatch");
            }
            for (int i = 0; i < calls.size(); i++) {
              calls.get(i).resolve(om, multiCallResult.get(i));
            }
            return null;
          } catch (JsonProcessingException e) {
            throw new RuntimeException("Invalid response, cause: decoding JSON error", e);
          }
        }));
      }

      class MyCall<T> implements Supplier<T> {

        private final AuthenticatedRequest<?> in;
        private final JavaType outT;
        private boolean executed = false;
        private T out;

        private MyCall(AuthenticatedRequest<?> in, JavaType outT) {
          this.in = in;
          this.outT = outT;
        }

        @Override
        public T get() {
          if (!executed) throw new IllegalStateException("Multi-call builder not executed");
          return out;
        }

        private void resolve(ObjectMapper om, JsonNode node) throws JsonProcessingException {
          out = om.treeToValue(node, outT);
          executed = true;
        }
      }
    };
  }

  public void disconnect() {
    if (invoker.get() != null) {
      log.debug("Disconnecting API ...");
      invoker.get().disconnect();
    }
  }

  @Override
  public void close() {
    disconnect();
  }

  protected ServerInvoker buildServerInvoker(String url, Integer timeout, String servicePath,
      CloseableHttpClient httpClient) {
    return new ServerInvoker(url, timeout, servicePath, httpClient);
  }

  private String buildServerPath() {
    String newPath = PROTOCOL + auth.get().getPath();
    newPath = newPath.endsWith("/") ? newPath.substring(0, newPath.length() - 1) : newPath;
    return newPath;
  }
}
