/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.http.invoker;

import static org.apache.hc.core5.http.io.entity.HttpEntities.create;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.geotab.api.Api;
import com.geotab.http.exception.ErrorHandler;
import com.geotab.http.exception.ResponseFailException;
import com.geotab.http.request.BaseRequest;
import com.geotab.http.response.BaseResponse;
import com.geotab.model.error.JsonRpcError;
import com.geotab.model.serialization.ApiJsonSerializer;
import com.geotab.util.Util.FailableFunction;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Optional;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.HttpException;
import org.apache.hc.core5.http.HttpStatus;
import org.apache.hc.core5.util.Timeout;

/**
 * A class for communicating over the Internet to MyGeotab.
 */
@Slf4j
public class ServerInvoker {

  public static final int DEFAULT_TIMEOUT = 300000;
  public static final String DEFAULT_SERVICE_PATH = "apiv1";

  @Getter
  private String url;
  @Getter
  private Integer timeout;
  @Getter
  private String servicePath;
  @Getter
  private CloseableHttpClient httpClient;

  public ServerInvoker(String url) {
    this(url, DEFAULT_TIMEOUT, DEFAULT_SERVICE_PATH);
  }

  public ServerInvoker(String url, Integer timeout, String servicePath) {
    this(url, timeout, servicePath, null);
  }

  public ServerInvoker(String url, Integer timeout, String servicePath, CloseableHttpClient httpClient) {
    this.servicePath = servicePath;
    setUrl(url);
    this.timeout = Optional.ofNullable(timeout).orElse(DEFAULT_TIMEOUT);
    this.httpClient = Optional.ofNullable(httpClient).orElse(buildDefaultHttpClient());
    log.debug("ServerInvoker params: \n url = {}  \n timeout = {} \n {}",
        this.url, this.timeout, httpClient != null ? "custom http client" : "default http client");
  }

  private ObjectMapper om() {
    return ApiJsonSerializer.getInstance().getObjectMapper();
  }

  public void setUrl(String url) {
    log.debug("ServerInvoker url set to {}", url);
    this.url = url + "/" + Optional.ofNullable(this.servicePath).orElse(DEFAULT_SERVICE_PATH);
  }

  public <O extends BaseResponse<T>, T> Optional<T> invoke(BaseRequest<?> in, Class<O> outT) {
    return invokeUnsafe(in, om().constructType(outT));
  }

  public <T> Optional<T> invoke(BaseRequest<?> in, Api.MethodDescriptor<?, T> method) {
    return invokeUnsafe(in, TypeFactory.defaultInstance().constructParametricType(BaseResponse.class, method.resultType()));
  }

  public <O extends BaseResponse<T>, T> Optional<T> invokeUnsafe(BaseRequest<?> in, JavaType outT) {
    return Optional.ofNullable(doInvoke(url, in, inputStream -> {
      O response = om().readValue(inputStream, outT);
      if (response.getError() != null) {
        ErrorHandler.checkForError(in.getMethod(), response.getError());
      }
      return response;
    })).map(BaseResponse::getResult);
  }

  /** Note that this is a bit slower than previous invokes because it creates an intermediary JSON tree. */
  public Optional<JsonNode> invokeJson(BaseRequest<?> in) {
    return Optional.ofNullable(doInvoke(url, in, inputStream -> {
      JsonNode response = om().readTree(inputStream);
      if (response.hasNonNull("error")) {
        ErrorHandler.checkForError(in.getMethod(), om().treeToValue(response.get("error"), JsonRpcError.class));
      }
      return response;
    }).get("result"));
  }

  private <T> T doInvoke(String url, BaseRequest<?> in, FailableFunction<InputStream, T, IOException> decoder) {
    HttpPost httpPost = new HttpPost(url);
    try {
      httpPost.setEntity(create(om().writeValueAsString(in), ContentType.APPLICATION_JSON));
    } catch (JsonProcessingException ex) {
      throw new RuntimeException("Encoding JSON error", ex);
    }

    // add custom headers
    if (in.getHttpHeaders() != null && !in.getHttpHeaders().isEmpty()) {
      for (Map.Entry<String, Object> header : in.getHttpHeaders().entrySet()) {
        httpPost.setHeader(header.getKey(), header.getValue());
      }
    }

    try {
      return httpClient.execute(httpPost, res -> {
        if (res == null) {
          log.error("Got null response while calling {} with request {}", url, in);
          throw new HttpException("Unexpected empty response from " + url);
        }

        try (HttpEntity httpEntity = res.getEntity(); InputStream inputStream = httpEntity.getContent()) {
          int sc = res.getCode();
          if (/*redirect*/sc == HttpStatus.SC_MOVED_TEMPORARILY || sc == HttpStatus.SC_MOVED_PERMANENTLY) {
            String redirectUrl = res.getHeader("Location").getValue();
            log.info("Got redirect response to {}", redirectUrl);
            try {
              return doInvoke(redirectUrl, in, decoder);
            } catch (Exception ex) {
              log.error("Can not redirect call to url {}", redirectUrl);
              String msg = "Can not redirect call to url " + redirectUrl;
              throw new HttpException(msg, new ResponseFailException(redirectUrl, sc, ex.getMessage(), ex));
            }
          } else if (/*failed*/sc != HttpStatus.SC_OK && sc != HttpStatus.SC_PARTIAL_CONTENT) {
            try {
              String msg = om().readValue(inputStream, String.class);
              log.debug("Unsuccessful response (code {}) : \n{}", sc, msg);
              throw new HttpException(msg, new ResponseFailException(url, sc, msg, null));
            } catch (Exception ex) {
              log.debug("Can not read unsuccessful response (code {})", sc, ex);
              throw new HttpException(ex.getMessage(), new ResponseFailException(url, sc, ex.getMessage(), ex));
            }
          } else /*succeed*/ {
            return decoder.apply(inputStream);
          }
        }
      });
    } catch (IOException ex) {
      throw new RuntimeException("Invoking request error", ex);
    }
  }

  private CloseableHttpClient buildDefaultHttpClient() {
    return HttpClients.custom()
        .disableAutomaticRetries()
        .disableRedirectHandling()
        .setDefaultRequestConfig(RequestConfig.custom()
            .setConnectTimeout(Timeout.ofSeconds(timeout))
            .setResponseTimeout(Timeout.ofSeconds(timeout))
            .build())
        .build();
  }

  /**
   * Closes httpClient's connections.
   */
  public void disconnect() {
    try {
      log.debug("Disconnecting http client from {} ...", url);
      if (httpClient != null) {
        httpClient.close();
      }
      log.info("Disconnected http client from {}", url);
    } catch (IOException e) {
      log.error("Can not disconnect http client from {}", url, e);
    }
  }
}
