/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.serialization.ByteArrayDeserializer;
import com.geotab.model.serialization.ByteArraySerializer;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * A custom parameter extends the parameters already supported by MyGeotab. This enables one to add a parameter to
 * MyGeotab that is not explicitly supported in the user interface.
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
@Builder
public class CustomParameter {

  /**
   * The byte values of the parameter starting at offset.
   */
  @JsonDeserialize(using = ByteArrayDeserializer.class)
  @JsonSerialize(using = ByteArraySerializer.class)
  private Byte[] bytes;

  /**
   * The description of {@link CustomParameter} that is displayed to the user. It does not in any way relate to the data
   * that is actually stored in the parameter itself.
   */
  private String description;

  /**
   * A value indicating whether enables or disables this parameter. If disabled, the parameter won't be sent and the
   * values 0 will be sent in the buffer.
   */
  @JsonProperty("isEnabled")
  private boolean isEnabled;

  /**
   * The offset into the parameter area where this {@link CustomParameter} will be written to.
   */
  private Integer offset;

  /**
   * Returns [true] if the custom parameter affects a bitmap options byte or [false] otherwise.
   *
   * @param parameter The parameter.
   * @return true | false
   */
  public static boolean isBitMappedOptionParameter(CustomParameter parameter) {
    if (parameter.getBytes().length == 1) {
      switch (parameter.getOffset()) {
        case 11: // Options
        case 12: // Options
        case 13: // Options
        case 14: // Options
        case 58: // Options
        case 63: // Options
        case 64: // Options
        case 65: // Options
        case 66: // Options
          return true;
        default:
          return false;
      }
    }
    return false;
  }

  /**
   * Check if "Reset Driver ID at Ignition Off" parameter is set.
   *
   * @param parameters The parameters.
   * @return true | false
   */
  public static boolean isResetDriverIdAtIgnitionOff(List<CustomParameter> parameters) {
    if (parameters == null || parameters.size() == 0) {
      return false;
    }
    for (CustomParameter customParameter : parameters) {
      if (customParameter.isEnabled()
          && customParameter.getBytes().length == 1
          && customParameter.getBytes()[0] == 16) {
        return true;
      }
    }
    return false;
  }

  /**
   * Checks whether a parameter is a Driver Blacklist parameter.
   *
   * @param parameter The parameter to check.
   * @return true if the {@link CustomParameter} is a Driver Blacklist parameter or false
   */
  public static boolean isDriverBlacklistParameter(CustomParameter parameter) {
    return parameter.getOffset() >= 94 && parameter.getOffset() <= 110;
  }

  /**
   * Validates that the given set of parameters is valid (no overlaps and doesn't collide with known parameters.
   *
   * @param parameters The parameters.
   * @return true if valid, false otherwise.
   */
  public static boolean isNotOverlapping(List<CustomParameter> parameters) {
    for (CustomParameter customParameter : parameters) {
      if (customParameter.isEnabled()) {
        for (CustomParameter compareTo : parameters) {
          if (customParameter.equals(compareTo)) {
            continue;
          }
          if (compareTo.isEnabled()) {
            if (isBitMappedOptionParameter(customParameter)) {
              if (customParameter.getOffset().equals(compareTo.getOffset())
                  && customParameter.getBytes()[0].equals(compareTo.getBytes()[0])) {
                return false;
              }
            } else if (customParameter.getOffset()
                <= compareTo.getOffset() + compareTo.getBytes().length - 1
                && customParameter.getOffset() + customParameter.getBytes().length - 1 >= compareTo
                .getOffset()) {
              return false;
            }
          }
        }
      }
    }
    return true;
  }

  @Override
  public boolean equals(Object object) {
    if (!(object instanceof CustomParameter)) {
      return false;
    }

    CustomParameter customParameter = (CustomParameter) object;

    return Arrays.equals(bytes, customParameter.getBytes())
        && Objects.equals(description, customParameter.getDescription())
        && isEnabled == customParameter.isEnabled()
        && offset.equals(customParameter.getOffset());
  }

  @Override
  public int hashCode() {
    int hash = 823118405;
    if (bytes != null) {
      for (Byte b : bytes) {
        hash ^= b.hashCode();
      }
    }
    hash = hash ^ (description != null ? description : "").hashCode()
        ^ Boolean.valueOf(isEnabled).hashCode()
        ^ offset.hashCode();
    return hash;
  }
}
