/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.device;

import static com.geotab.model.entity.device.Device.A1_PREFIX;
import static com.geotab.model.entity.device.Device.A1_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.CUSTOM_DEVICE_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO10_PREFIX;
import static com.geotab.model.entity.device.Device.GO10_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO2_PREFIX;
import static com.geotab.model.entity.device.Device.GO2_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO3_PREFIX;
import static com.geotab.model.entity.device.Device.GO3_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO4_PREFIX;
import static com.geotab.model.entity.device.Device.GO4_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO4_V3_PREFIX;
import static com.geotab.model.entity.device.Device.GO4_V3_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO5_PREFIX;
import static com.geotab.model.entity.device.Device.GO5_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO6_PREFIX;
import static com.geotab.model.entity.device.Device.GO6_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO7_PREFIX;
import static com.geotab.model.entity.device.Device.GO7_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO8_PREFIX;
import static com.geotab.model.entity.device.Device.GO8_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO9B_PREFIX;
import static com.geotab.model.entity.device.Device.GO9B_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO9_PREFIX;
import static com.geotab.model.entity.device.Device.GO9_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.GO_DRIVE_PREFIX;
import static com.geotab.model.entity.device.Device.GO_DRIVE_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.OLD_GEOTAB_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.U1_PREFIX;
import static com.geotab.model.entity.device.Device.U1_PRODUCT_ID;
import static com.geotab.model.entity.device.Device.UNTRACKED_ASSET_PRODUCT_ID;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import java.util.function.Supplier;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum DeviceType implements HasName {

  /**
   * Unknown device type.
   */
  @JsonEnumDefaultValue
  NONE(DeviceTypeNameConstants.NONE, "", UNTRACKED_ASSET_PRODUCT_ID, UntrackedAsset.class, UntrackedAsset::new),

  /**
   * GEOTAB unit (unsupported).
   */
  OLD_GEOTAB(DeviceTypeNameConstants.OLD_GEOTAB, GO2_PREFIX, OLD_GEOTAB_PRODUCT_ID, GoLegacy.class, GoLegacy::new),

  /**
   * GO Device.
   */
  GO2(DeviceTypeNameConstants.GO2, GO2_PREFIX, GO2_PRODUCT_ID, GoLegacy.class, GoLegacy::new),

  /**
   * GO3 Tracking Device.
   */
  GO3(DeviceTypeNameConstants.GO3, GO3_PREFIX, GO3_PRODUCT_ID, GoLegacy.class, GoLegacy::new),

  /**
   * GO4 Tracking Device.
   */
  GO4(DeviceTypeNameConstants.GO4, GO4_PREFIX, GO4_PRODUCT_ID, GoLegacy.class, GoLegacy::new),

  /**
   * GO4v3 Tracking Device - like GO4 but with larger memory and some new functionality.
   */
  GO4V3(DeviceTypeNameConstants.GO4V3, GO4_V3_PREFIX, GO4_V3_PRODUCT_ID, Go4v3.class, Go4v3::new),

  /**
   * GO5 Tracking Device.
   */
  GO5(DeviceTypeNameConstants.GO5, GO5_PREFIX, GO5_PRODUCT_ID, Go5.class, Go5::new),

  /**
   * GO6 Tracking Device.
   */
  GO6(DeviceTypeNameConstants.GO6, GO6_PREFIX, GO6_PRODUCT_ID, Go6.class, Go6::new),

  /**
   * GO7 Tracking Device.
   */
  GO7(DeviceTypeNameConstants.GO7, GO7_PREFIX, GO7_PRODUCT_ID, Go7.class, Go7::new),

  /**
   * GO8 Tracking Device.
   */
  GO8(DeviceTypeNameConstants.GO8, GO8_PREFIX, GO8_PRODUCT_ID, Go8.class, Go8::new),

  /**
   * GO9 Tracking Device.
   */
  GO9(DeviceTypeNameConstants.GO9, GO9_PREFIX, GO9_PRODUCT_ID, Go9.class, Go9::new),

  /**
   * GO9 Tracking Device.
   */
  GO9B(DeviceTypeNameConstants.GO9B, GO9B_PREFIX, GO9B_PRODUCT_ID, Go9b.class, Go9b::new),

  /**
   * GO10 Tracking Device.
   */
  GO10(DeviceTypeNameConstants.GO10, GO10_PREFIX, GO10_PRODUCT_ID, Go10.class, Go10::new),

  /**
   * Custom (third-party) device.
   */
  CUSTOM_DEVICE(DeviceTypeNameConstants.CUSTOM_DEVICE, "", CUSTOM_DEVICE_PRODUCT_ID, CustomDevice.class,
      CustomDevice::new),

  /**
   * GoDrive Mobile Device.
   */
  GO_DRIVE_DEVICE(DeviceTypeNameConstants.GO_DRIVE_DEVICE, GO_DRIVE_PREFIX, GO_DRIVE_PRODUCT_ID, GoDriveDevice.class,
      GoDriveDevice::new),

  /**
   * Custom Vehicle Device.
   */
  CUSTOM_VEHICLE_DEVICE(DeviceTypeNameConstants.CUSTOM_VEHICLE_DEVICE, "", 0, CustomVehicleDevice.class,
      CustomVehicleDevice::new),

  /**
   * A1 Tracking Device.
   */
  A1(DeviceTypeNameConstants.A1, A1_PREFIX, A1_PRODUCT_ID, A1.class, A1::new),

  /**
   * A1 Tracking Device.
   */
  U1(DeviceTypeNameConstants.U1, U1_PREFIX, U1_PRODUCT_ID, U1.class, U1::new),;

  private final String name;
  private final String prefix;
  private final int productId;
  private final Class<? extends Device> type;
  private final Supplier<Device> constructor;

  DeviceType(String name, String prefix, int productId, Class<? extends Device> type, Supplier<Device> constructor) {
    this.name = name;
    this.prefix = prefix;
    this.productId = productId;
    this.type = type;
    this.constructor = constructor;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public String getPrefix() {
    return prefix;
  }

  public int getProductId() {
    return productId;
  }

  public Class<? extends Device> getType() {
    return type;
  }

  public Device newInstance() {
    return constructor.get();
  }

  public static DeviceType findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          DeviceType.class.getSimpleName(), DeviceType.NONE);
      return NONE;
    }

    DeviceType deviceType = findByName(name);
    if (deviceType != null) {
      return deviceType;
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, DeviceType.class.getSimpleName(), DeviceType.NONE);
    return NONE;
  }

  public static DeviceType findByName(String name) {
    if (Util.isEmpty(name)) {
      return null;
    }

    for (DeviceType deviceType : values()) {
      if (deviceType.getName().equalsIgnoreCase(name.trim())) {
        return deviceType;
      }
    }

    return null;
  }
}

