package com.geotab.model.entity.device;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.CustomParameter;
import com.geotab.model.entity.enginetype.EngineType;
import com.geotab.model.entity.group.Group;
import com.geotab.model.serialization.EntityCollectionAsIdCollectionSerializer;
import com.geotab.model.wifi.WifiUsageTier;
import com.geotab.util.DeviceDefaultsUtil;
import java.util.Arrays;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * The base device for all Geotab GO products. Additional properties can be seen in {@link Device}.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public abstract class GoDevice extends Device implements VehicleIdentificationNumberAware, LicensableAware {

  /**
   * Gets or sets the messaging status {@link Group}(s). Drivers with Garmin-equipped vehicles can update their working
   * status by choosing one from a set of common statuses. This status update is shared with their team. For example,
   * when drivers finish their work, they can set their statuses to 'Available'. Their dispatcher is notified of this
   * and can begin to assign work based on both the location and availability of the drivers. Default [Empty].
   */
  @JsonSerialize(converter = EntityCollectionAsIdCollectionSerializer.class)
  protected List<Group> autoGroups;

  /**
   * The set of CustomParameter(s) associated with this device. Custom parameters allow the activation of special
   * features — limited to custom and beta firmware. Custom parameters are issued only when necessary. Default [Empty].
   */
  private List<CustomParameter> customParameters;

  /**
   * Gets or sets master toggle to disable the device buzzer. When set to [true], the device will not provide driver
   * feedback of any kind. Default [false].
   */
  private boolean disableBuzzer;

  /**
   * Gets or sets toggle to enable beeping when the vehicle idles for more than IdleMinutes. Default [false].
   */
  private boolean enableBeepOnIdle;

  /**
   * Gets or sets flag to force the parameters to be updated between the store and a go device, the store will be
   * updated with the parameter version of the go device +1. This is to avoid the issue where: a device is added (store
   * parameter v0, device parameter v0), parameter version has changed (store parameter v1, device parameter v1), device
   * has been removed, device is re-added (store parameter v0, device parameter v1). Thus when the store is updated to
   * v1 there would be no change.
   */
  private boolean enableMustReprogram;

  /**
   * Gets or sets toggle to enable speed warning value for the vehicle. When enabled [true], only beep briefly (instead
   * of continuously), when 'SpeedingOn' value is exceeded. 'IsSpeedIndicator' must also be enabled. Default [false].
   */
  private boolean enableSpeedWarning;

  /**
   * Gets or sets the EngineType. Default [EngineTypeGeneric].
   *
   * <p>Deprecated. The EngineType entity will be removed completely in a future version of the
   * SDK.
   */
  private EngineType engineType;

  /**
   * Gets or sets the Vehicles Identity Number reported by the engine.
   */
  private String engineVehicleIdentificationNumber;

  /**
   * Gets or sets a value indicating whether to wake up the GPS while the vehicle is parked: this will allow for a
   * faster GPS latch when the vehicle begins its trip. Note: This will drain the vehicle's battery at a faster rate and
   * should not be used with newer devices. Default [false].
   */
  private boolean ensureHotStart;

  /**
   * Gets or sets the GPS off delay in minutes. When enabled this allows the device to keep the GPS on for a period
   * after the vehicle has been turned off. Normally, the GPS turns off immediately. Keeping the GPS on can improve
   * tracking on older devices when many stops are made.
   */
  private Integer gpsOffDelay;

  /**
   * Gets or sets the number of minutes of allowed idling before device beeping starts. EnableBeepOnIdle must be
   * enabled. Default [3].
   */
  private Integer idleMinutes;


  /**
   * Gets or sets a toggle to beep constantly when the vehicle reaches the speed set in 'SpeedingOn', and do not stop
   * until the vehicle slows below the 'SpeedingOff' speed. To only beep briefly (instead of continuously), enable
   * 'EnableSpeedWarning'. Default [false].
   */
  private Boolean isSpeedIndicator;

  /**
   * Gets or sets the vehicle license plate details of the vehicle associated with the device. Maximum length [50]
   * Default [""].
   */
  private String licensePlate;

  /**
   * Gets or sets the state or province of the vehicle associated with the device. Maximum length [50] Default [""].
   */
  private String licenseState;

  /**
   * Gets or sets the device major firmware version. Newer versions have more functionality. Live device firmware
   * versions are managed automatically. Default [0].
   */
  private Short major;

  /**
   * Gets or sets the minimum accident speed in km/h. Default [4].
   */
  private Double minAccidentSpeed;

  /**
   * Gets or sets the device minor firmware version. Newer versions have more functionality. Live device firmware
   * versions are managed automatically. Default [0].
   */
  private Short minor;

  /**
   * Gets or sets the parameter version that is stored in MyGeotab. The parameter version should be increased by one
   * when the parameters have been modified and need to be synced with the physical device. Default [0].
   */
  private Integer parameterVersion;

  /**
   * Gets or sets a toggle that determines whether or not a device is under automatic vehicle management. Default
   * [false].
   */
  private Boolean pinDevice;

  /**
   * Gets or sets the speeding off value in km/h. When 'IsSpeedIndicator' is enabled, once beeping starts, the vehicle
   * must slow down to this speed for the beeping to stop. Default [90].
   */
  private Double speedingOff;

  /**
   * Gets or sets the speeding on value in km/h. When 'IsSpeedIndicator' is enabled, the device will start beeping when
   * the vehicle exceeds this speed. Default [100].
   */
  private Double speedingOn;

  /**
   * Gets or sets the Vehicle Identification Number (VIN) of the vehicle associated with the device. Maximum length [50]
   * Default [""].
   */
  private String vehicleIdentificationNumber;

  /**
   * Gets or sets the {@link GoDevice} of an attached GoTalk. Default [English].
   */
  private GoTalkLanguage goTalkLanguage;

  /**
   * Gets or sets the capacity of all usable fuel tanks in litres. Default [0].
   */
  private Double fuelTankCapacity;

  private Boolean disableSleeperBerth;

  /**
   * Gets or sets a toggle that represents automatic generation of DutyStatusLog's for a {@link Device}.
   */
  private HosOption autoHos;

  /**
   * The list of Wifi Usage Tiers for Wifi Hotspot rate plans. Will only be populated if this device has a RatePlan
   * which has Wifi Hotspot functionality.
   */
  private List<WifiUsageTier> wifiHotspotLimits;

  @Override
  public Boolean isPinDevice() {
    return getPinDevice();
  }

  /**
   * Gets returns true if this device supports external device power control. Originally only GO4v3 was to support this
   * - but there are a significant number of customers that had GO3V2s with this feature. As per DB: We had to support
   * Iridium on Go4v2 for Oz and Garmin on GO4v2 for Toronto Hydro 78.x.x and 79.x.x supports Iridium, 66.x.x (J1708
   * Live) and 80.x.x supports Garmin.
   */
  @JsonIgnore
  public Boolean isExternalDevicePowerControlSupported() {
    if (deviceType == null || productId == null) {
      return null;
    }
    return deviceType == DeviceType.GO9
        || deviceType == DeviceType.GO9B
        || deviceType == DeviceType.GO8
        || deviceType == DeviceType.GO7
        || deviceType == DeviceType.GO6
        || deviceType == DeviceType.GO4V3
        || productId == 66
        || (productId >= 78 && productId <= 80);
  }

  /**
   * Converts bytes to degrees.
   *
   * @param value The value.
   * @return The angle.
   */
  public static int byteToDegrees(short value) {
    return (int) Math.round(2.403 * value);
  }

  /**
   * Converts byte to speed.
   *
   * @param value The value.
   * @return The speed in km/h.
   */
  public static double byteToSpeed(short value) {
    return value * 3.6 * 0.512;
  }


  /**
   * Checks the provided array is the correct length and pads it to the correct length if it is not.
   *
   * @param booleanArray The boolean array.
   * @return Array of boolean.
   */
  public static boolean[] checkAndPadAuxArray(boolean[] booleanArray) {
    if (booleanArray == null) {
      return new boolean[8];
    }
    if (booleanArray.length > 8) {
      throw new IllegalArgumentException("booleanArray.length cannot be more then 8");
    }
    return Arrays.copyOf(booleanArray, 8);
  }

  /**
   * Checks the provided array is the correct length and pads it to the correct length if it is not.
   *
   * @param doubleArray The double array.
   * @return Array of double.
   */
  public static double[] checkAndPadAuxArray(double[] doubleArray) {
    if (doubleArray == null) {
      return new double[8];
    }
    if (doubleArray.length > 8) {
      throw new IllegalArgumentException("doubleArray.length cannot be more then 8");
    }
    return Arrays.copyOf(doubleArray, 8);
  }

  /**
   * Checks the provided array is the correct length and pads it to the correct length if it is not.
   *
   * @param booleanArray The boolean array.
   * @return Array of boolean.
   */
  public static boolean[] checkAndPadAuxIgnArray(boolean[] booleanArray) {
    if (booleanArray == null) {
      return new boolean[4];
    }
    if (booleanArray.length > 4) {
      throw new IllegalArgumentException("booleanArray.Length cannot be more then 4");
    }
    return Arrays.copyOf(booleanArray, 4);
  }

  /**
   * Checks the provided array is the correct length and pads it to the correct length if it is not.
   *
   * @param shortArray The short array.
   * @return Array of short.
   */
  public static short[] checkAndPadChannelArray(short[] shortArray) {
    if (shortArray == null) {
      return new short[20];
    }
    if (shortArray.length > 20) {
      throw new IllegalArgumentException("shortArray.length cannot be more then 20");
    }
    return Arrays.copyOf(shortArray, 20);
  }

  /**
   * Converts Degrees to byte.
   *
   * @param value The value.
   * @return The byte representation of the direction.
   */
  public static short degreesToByte(int value) {
    return (short) (int) Math.round(value / 2.403);
  }

  /**
   * Converts speed to byte.
   *
   * @param value The value.
   * @return The byte representation of the speed.
   */
  public static short speedToByte(double value) {
    return (short) (int) Math.round(value / 3.6 / 0.512);
  }

  /**
   * Determines whether garmin is supported on this device.
   *
   * @return Boolean
   */
  @JsonIgnore
  public Boolean isGarminSupported() {
    if (deviceType == null || productId == null) {
      return null;
    }
    return deviceType == DeviceType.GO9
        || deviceType == DeviceType.GO9B
        || deviceType == DeviceType.GO8
        || deviceType == DeviceType.GO7
        || deviceType == DeviceType.GO6
        || deviceType == DeviceType.GO4V3
        || productId == 66
        || productId == 80;
  }

  /**
   * Returns true if this device is definitely an rf device. Device will need to connect once before the product id is
   * available to check.
   */
  @SuppressWarnings("AbbreviationAsWordInName")
  @JsonIgnore
  public Boolean isRFDevice() {
    if (productId == null) {
      return null;
    }
    switch (productId) {
      case 3:
      case 5:
      case 10:
      case 12:
      case 14:
      case 32:
      case 36:
      case 38:
      case 39:
      case 68:
      case 69:
      case 70:
      case 75:
      case 76:
      case 77:
      case 78:
      case 82:
      case 84: {
        return true;
      }
      case 17:
      case 20:
      case 21:
      case 26:
      case 27:
      case 30:
      case 34:
      case 35:
      case 37:
      case 65:
      case 66:
      case 67:
      case 71:
      case 72:
      case 73:
      case 79:
      case 80:
      case 81:
      case 83: {
        return false;
      }
      default: {
        return deviceType == DeviceType.GO5
            || deviceType == DeviceType.GO6
            || deviceType == DeviceType.GO7
            || deviceType == DeviceType.GO8
            || deviceType == DeviceType.GO9
            || deviceType == DeviceType.GO9B
            || deviceType == DeviceType.CUSTOM_DEVICE
            || deviceType == DeviceType.CUSTOM_VEHICLE_DEVICE
            || deviceType == DeviceType.GO_DRIVE_DEVICE ? false : null;
      }
    }
  }

  /**
   * Gets or sets the parameter version that is on the device. Can be used to track the parameter version currently on
   * the device by comparing to ParameterVersion. Default [0].
   */
  private Short parameterVersionOnDevice;

  @Override
  public void populateDefaults() {
    super.populateDefaults();
    DeviceDefaultsUtil.addVehicleIdentificationNumberDefaults(this);
    DeviceDefaultsUtil.addGoDeviceDefaults(this);
  }
}
