package com.geotab.model.entity.device;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.geotab.util.DeviceDefaultsUtil;
import java.util.Optional;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * A legacy Geotab device is any GO device prior to GO4v3. Additional properties can be seen in {@link Device}.
 */
@Data
@NoArgsConstructor
@SuperBuilder(builderMethodName = "goLegacyBuilder")
public class GoLegacy extends GoDevice implements DigitalAuxiliariesAware, RfChannelsAware {

  /**
   * The distance to be used (in meters) to generate a log. A log will be reported each time the this distance has been
   * exceeded. Default [1000].
   */
  protected Integer deltaDistance;

  /**
   * The delta heading. Default [48].
   */
  protected Integer deltaHeading;

  /**
   * The delta heading HS. Default [14].
   */
  protected Integer deltaHeadingHS;

  /**
   * The delta heading min speed in km/h. Default [14].
   */
  protected Double deltaHeadingMinSpeed;

  /**
   * The delta heading min speed HS in km/h. Default [72].
   */
  protected Double deltaHeadingMinSpeedHS;

  /**
   * The delta min speed in km/h. Default [50].
   */
  protected Double deltaMinSpeed;

  /**
   * The delta speed in km/h. Default [9].
   */
  protected Double deltaSpeed;

  /**
   * The harsh break. Default [17].
   */
  protected Double harshBreak;

  /**
   * Whether to send warning on harsh brake. Default [false].
   */
  @JsonProperty("isHarshBrakeWarningOn")
  protected boolean isHarshBrakeWarningOn;

  /**
   * An array that can get/set the AuxWarningSpeeds for the vehicle.
   */
  protected double[] auxWarningSpeed;

  /**
   * An array that can get/set the EnableAuxWarnings for the vehicle.
   */
  protected boolean[] enableAuxWarning;

  /**
   * The EnableControlExternalRelay value for the vehicle.
   */
  protected boolean enableControlExternalRelay;

  /**
   * The External Device ShutDown Delay value for the vehicle.
   */
  protected int externalDeviceShutDownDelay;

  /**
   * The immobilize arming.
   */
  protected int immobilizeArming;

  /**
   * The immobilize unit.
   */
  protected boolean immobilizeUnit;

  /**
   * An array that can get/set the IsAuxIgnTriggers for the vehicle.
   */
  @JsonProperty("isAuxIgnTrigger")
  protected boolean[] isAuxIgnTrigger;

  /**
   * An array of values indicating if a corresponding Aux signal should be inverted on importing the device data.
   */
  @JsonProperty("isAuxInverted")
  protected boolean[] isAuxInverted;

  /**
   * Exposes the radio channels as an array of 20 bytes.
   */
  protected short[] channel;

  /**
   * The channel count.
   */
  protected Byte channelCount;

  /**
   * The FrequencyOffset used on the device radio. The default is 1 and should not be changed. Only has an effect on
   * Channel Set 0 - which consists of channels 0 to 15.
   */
  protected Byte frequencyOffset;

  /**
   * A value indicating whether this instance is aided GPS enabled.
   */
  @JsonProperty("isAidedGpsEnabled")
  protected boolean isAidedGpsEnabled;

  /**
   * The IsRfUploadOnWhenMoving value for the vehicle.
   */
  @JsonProperty("isRfUploadOnWhenMoving")
  protected boolean isRfUploadOnWhenMoving;

  /**
   * The version of this set of rf channels. Each time a change is made, increment this value until it rolls over.
   */
  protected short rfParameterVersion;

  @Override
  public void populateDefaults() {
    super.populateDefaults();
    this.productId = Optional.ofNullable(this.productId).orElse(GO2_PRODUCT_ID);
    DeviceDefaultsUtil.addGoLegacyDefaults(this);
    DeviceDefaultsUtil.addRfChannelDefaults(this);
    DeviceDefaultsUtil.addAuxiliaryDefaults(this);
  }

  public GoLegacy setChannel(short[] channel) {
    this.channel = checkAndPadChannelArray(channel);
    return this;
  }

  public GoLegacy setEnableAuxWarning(boolean[] enableAuxWarning) {
    this.enableAuxWarning = checkAndPadAuxArray(enableAuxWarning);
    return this;
  }

  public GoLegacy setIsAuxIgnTrigger(boolean[] isAuxIgnTrigger) {
    this.isAuxIgnTrigger = checkAndPadAuxIgnArray(isAuxIgnTrigger);
    return this;
  }

  public GoLegacy setIsAuxInverted(boolean[] isAuxInverted) {
    this.isAuxInverted = checkAndPadAuxArray(isAuxInverted);
    return this;
  }

  public GoLegacy setAuxWarningSpeed(double[] auxWarningSpeed) {
    this.auxWarningSpeed = checkAndPadAuxArray(auxWarningSpeed);
    return this;
  }
}
