/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.device;

import com.geotab.util.DeviceDefaultsUtil;
import java.util.Optional;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * An untracked asset that is used in MyGeotab without a serial number. This is used for extensibility and is based on
 * the {@link Device} object.
 */
@Data
@NoArgsConstructor
@SuperBuilder(builderMethodName = "untrackedAssetBuilder")
public class UntrackedAsset extends Device implements OdometerAdjustmentProviderAware,
    EngineHoursAdjustmentProviderAware, LicensableAware {

  /**
   * Vehicle Identification Number (VIN) of the vehicle associated with the device. Maximum length [50] Default [""].
   */
  private String vehicleIdentificationNumber;

  /**
   * Engine Vehicle Identification Number from the engine of the vehicle associated with the device.
   */
  private String engineVehicleIdentificationNumber;

  /**
   * The Odometer for the device.
   */
  private Double odometer;

  /**
   * Value used to correct the odometer value received from the engine. Default [1].
   */
  private Float odometerFactor;

  /**
   * The offset to be applied odometer reported by the engine computer. Default [0].
   */
  private Double odometerOffset;

  /**
   * The Engine Hours for the device. Default [0].
   */
  private Double engineHours;

  /**
   * The offset to be applied engine hours data reported by the engine computer. Default [0].
   */
  private Integer engineHourOffset;

  /**
   * The vehicle license plate details of the vehicle associated with the device. Maximum length [50] Default [""].
   */
  private String licensePlate;

  /**
   * The state or province of the vehicle associated with the device. Maximum length [50] Default [""].
   */
  private String licenseState;

  /**
   * Gets or sets a toggle that determines whether or not a device is under automatic vehicle management. Default
   * [false].
   */
  private Boolean pinDevice;

  @Override
  public void populateDefaults() {
    super.populateDefaults();
    productId = Optional.ofNullable(productId).orElse(UNTRACKED_ASSET_PRODUCT_ID);
    vehicleIdentificationNumber = Optional.ofNullable(vehicleIdentificationNumber).orElse("");
    engineVehicleIdentificationNumber = Optional.ofNullable(engineVehicleIdentificationNumber)
        .orElse("?");
    DeviceDefaultsUtil.addOdometerAdjustmentProviderDefaults(this);
    DeviceDefaultsUtil.addEngineHoursAdjustmentProviderDefaults(this);
    DeviceDefaultsUtil.addLicensableDefaults(this);
  }
}
