package com.geotab.model.entity.driverchange;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.user.Driver;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum DriverChangeType implements HasName {

  /**
   * Unknown key type.
   */
  @JsonEnumDefaultValue
  NONE("None", 0),

  /**
   * Driver Key - based Driver. Drivers use this key type to identify themselves. This will typically be used when the
   * driver ID relay is enabled. Makes a {@link com.geotab.model.entity.device.Device} continuously associated with a
   * particular {@link com.geotab.model.entity.user.Driver} until a different driver ID is assigned to the same device.
   */
  DRIVER_KEY("DriverKey", 1),

  /**
   * DriverVehicleChange log - based Driver. Auto-generated by the system to support the driver ID relay feature.
   */
  DRIVER_VEHICLE_CHANGE("DriverVehicleChange", 256),

  /**
   * User-based Driver that is only active for the duration of a single trip.
   */
  TRIP_DRIVER("TripDriver", 257),

  /**
   * User-based Driver. Makes a {@link Device} continuously associated with particular {@link Driver} until a different
   * driver id is assigned to the same device. First record in a trip overrides any other record encountered during the
   * trip it was added to except "TripDriver".
   */
  DRIVER("Driver", 258),

  /**
   * System-detected Driver reset (to UnknownDriver). Makes a {@link Device} permanently disassociate from the previous
   * {@link Driver} due to concurrent permanent registration of the same Driver for a different device.
   */
  RESET_DRIVER("ResetDriver", 259);

  private final String name;
  private final int code;

  DriverChangeType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public static DriverChangeType findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          DriverChangeType.class.getSimpleName(), DriverChangeType.NONE);
      return NONE;
    }

    for (DriverChangeType driverChangeType : values()) {
      if (driverChangeType.getName().equalsIgnoreCase(name.trim())) {
        return driverChangeType;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, DriverChangeType.class.getSimpleName(), DriverChangeType.NONE);
    return NONE;
  }
}
