/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.dutystatus;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum DutyStatusLogType implements HasName {

  /**
   * The none.
   */
  @JsonEnumDefaultValue
  NONE("None", -1),

  /**
   * On-duty status.
   */
  ON("ON", 0),

  /**
   * Off-duty status.
   */
  OFF("OFF", 1),

  /**
   * Drive status.
   */
  D("D", 2),

  /**
   * Sleeper berth status.
   */
  SB("SB", 3),

  /**
   * Exemption 16 hours.
   */
  EXEMPTION_16_H("Exemption16H", 4),

  /**
   * Adverse weather and driving conditions exemption.
   */
  ADVERSE_WEATHER("AdverseWeather", 5),

  /**
   * System log for device power connection.
   */
  CONNECTED("Connected", 6),

  /**
   * System log for device power disconnection.
   */
  DISCONNECTED("Disconnected", 7),

  /**
   * Personal conveyance driver status.
   */
  PC("PC", 8),

  /**
   * Yard move driver status.
   */
  YM("YM", 9),

  /**
   * Wait time oil well driver status.
   */
  WT("WT", 10),

  /**
   * Authority status.
   */
  AUTHORITY("Authority", 11),

  /**
   * User login record.
   */
  LOGIN("Login", 12),

  /**
   * User logout record.
   */
  LOGOFF("Logoff", 13),

  /**
   * Daily certify record.
   */
  CERTIFY("Certify", 14),

  /**
   * Engine power up record.
   */
  ENGINE_POWERUP("EnginePowerup", 15),

  /**
   * Engine shutdown record.
   */
  ENGINE_SHUTDOWN("EngineShutdown", 16),

  /**
   * Engine power up in PC record.
   */
  ENGINE_POWERUP_PC("EnginePowerupPC", 17),

  /**
   * Engine shutdown in PC record.
   */
  ENGINE_SHUTDOWN_PC("EngineShutdownPC", 18),

  /**
   * Engine power status engages ELD within 1 minute.
   */
  POWER_COMPLIANCE("PowerCompliance", 19),

  /**
   * Occurs when engine information (power, motion, miles, and hours) cannot be obtained by ELD.
   */
  ENGINE_SYNC_COMPLIANCE("EngineSyncCompliance", 20),

  /**
   * When ELD date and time exceeds 10 minute offset from UTC.
   */
  TIMING_COMPLIANCE("TimingCompliance", 21),

  /**
   * ELD continually fails to acquire valid position measurement.
   */
  POSITIONING_COMPLIANCE("PositioningCompliance", 22),

  /**
   * Storage capacity is reached, or missing data elements exist.
   */
  DATA_RECORDING_COMPLIANCE("DataRecordingCompliance", 23),

  /**
   * Transfer of data fails to complete.
   */
  DATA_TRANSFER_COMPLIANCE("DataTransferCompliance", 24),

  /**
   * Other instances of DutyStatusMalfunctionTypes.Malfunction or
   * <br>DutyStatusMalfunctionTypes.Diagnostic.
   */
  OTHER_COMPLIANCE("OtherCompliance", 25),

  /**
   * Missing data elements.
   */
  MISSING_ELEMENT_COMPLIANCE("MissingElementCompliance", 26),

  /**
   * More than 30 minutes of driving with unidentified driving.
   */
  UNIDENTIFIED_DRIVING_COMPLIANCE("UnidentifiedDrivingCompliance", 27),

  /**
   * Intermediate Personal Conveyance Event.
   */
  INT_PC("INT_PC", 28),

  /**
   * Intermediate Drive Event.
   */
  INT_D("INT_D", 29),

  /**
   * YM, PC, or WT clearing event.
   */
  SITUATIONAL_DRIVING_CLEAR("SituationalDrivingClear", 30),

  /**
   * Exemption off duty deferral.
   */
  EXEMPTION_OFF_DUTY_DEFERRAL("ExemptionOffDutyDeferral", 31),

  /**
   * Hos Enabled.
   */
  HOS_ENABLED("HosEnabled", 32),

  /**
   * Hos Disabled.
   */
  HOS_DISABLED("HosDisabled", 33),

  /**
   * Special type of PC, will not be interrupted by automatic logs.
   */
  PC_EXEMPTED("PC_Exempted", 34),

  /**
   * Canadian ELD: Cycle set to "Cycle 1".
   */
  CANADA_CYCLE_ONE("CanadaCycleOne", 35),

  /**
   * Canadian ELD: Cycle set to "Cycle 2".
   */
  CANADA_CYCLE_TWO("CanadaCycleTwo", 36),

  /**
   * Canadian ELD: Operating zone set to "south of latitude 60N in Canada".
   */
  OPERATING_ZONE_CANADA_SOUTH_OF_60("OperatingZoneCanadaSouthOf60", 37),

  /**
   * Canadian ELD: Operating zone set to "north of latitude 60N in Canada".
   */
  OPERATING_ZONE_CANADA_NORTH_OF_60("OperatingZoneCanadaNorthOf60", 38),

  /**
   * Canadian ELD: Operating zone set to "United States".
   */
  OPERATING_ZONE_AMERICA("OperatingZoneAmerica", 39),

  /**
   * Intermediate log for co-drivers for ELD compliance.
   */
  INT_CO_DRIVER("INT_CoDriver", 40),

  /**
   * Any tasks relating to the use of the vehicle.
   */
  WORK("Work", 41),

  /**
   * Driver is not working in relation to a fatigue-regulated heavy vehicle who are not driving under BFM, AFM or
   * exemption hours.
   */
  REST("Rest", 42),

  /**
   * Work Exemption (permitted personal activity) refers to any work done by a driver for personal and non-commercial
   * purposes.
   */
  WORK_EXEMPTION("WorkExemption", 43),

  /**
   * Railroad Exemption.
   */
  RAILROAD_EXEMPTION("RailroadExemption", 43);

  private final String name;
  private final int code;

  DutyStatusLogType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public static DutyStatusLogType findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          DutyStatusLogType.class.getSimpleName(), DutyStatusLogType.NONE);
      return NONE;
    }

    for (DutyStatusLogType dutyStatusLogType : values()) {
      if (dutyStatusLogType.getName().equalsIgnoreCase(name.trim())) {
        return dutyStatusLogType;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, DutyStatusLogType.class.getSimpleName(), DutyStatusLogType.NONE);
    return NONE;
  }
}
