/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.dutystatus;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum DutyStatusMalfunctionTypes implements HasName {

  /**
   * No malfunction or diagnostic present or cleared.
   */
  @JsonEnumDefaultValue
  NONE("None", 0),
  /**
   * In a diagnostic state.
   */
  DIAGNOSTIC("Diagnostic", 1),

  /**
   * In a malfunction state.
   */
  MALFUNCTION("Malfunction", 2),

  /**
   * User has cleared the malfunction or diagnostic.
   */
  USER_CLEAR("UserClear", 3),

  /**
   * User has cleared the diagnostic.
   */
  USER_DIAGNOSTIC_CLEAR("UserDiagnosticClear", DIAGNOSTIC.getCode() | USER_CLEAR.getCode()),

  /**
   * User has cleared the malfunction.
   */
  USER_MALFUNCTION_CLEAR("UserMalfunctionClear", MALFUNCTION.getCode() | USER_CLEAR.getCode()),

  /**
   * System has determined that the malfunction or diagnostic is cleared. Not exported to FMCSA.
   */
  SYSTEM_CLEAR("SystemClear", 8),

  /**
   * System has determined that the diagnostic is cleared. Not exported to FMCSA.
   */
  SYSTEM_DIAGNOSTIC_CLEAR("SystemDiagnosticClear", DIAGNOSTIC.getCode() | SYSTEM_CLEAR.getCode()),

  /**
   * System has determined that the malfunction or diagnostic is cleared and the vehicle was in motion.
   */
  SYSTEM_CLEAR_DRIVING("SystemClearDriving", 16),

  /**
   * System has determined that the diagnostic is cleared and the vehicle was in motion.
   */
  SYSTEM_DIAGNOSTIC_CLEAR_DRIVING("SystemDiagnosticClearDriving",
      DIAGNOSTIC.getCode() | SYSTEM_CLEAR_DRIVING.getCode()),

  /**
   * User has inputted a manual address for the log during a position compliance diagnostic event.
   */
  MANUAL_POSITION("ManualPosition", 32),

  /**
   * Combination of ManualPosition and Diagnostic.
   */
  DIAGNOSTIC_MANUAL_POSITION("DiagnosticManualPosition",
      DIAGNOSTIC.getCode() | MANUAL_POSITION.getCode()),

  /**
   * Combination of ManualPosition and Malfunction.
   */
  MALFUNCTION_MANUAL_POSITION("MalfunctionManualPosition",
      MALFUNCTION.getCode() | MANUAL_POSITION.getCode());

  private final String name;
  private final int code;

  DutyStatusMalfunctionTypes(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public static DutyStatusMalfunctionTypes findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          DutyStatusMalfunctionTypes.class.getSimpleName(), DutyStatusMalfunctionTypes.NONE);
      return NONE;
    }

    for (DutyStatusMalfunctionTypes dutyStatusMalfunctionTypes : values()) {
      if (dutyStatusMalfunctionTypes.getName().equalsIgnoreCase(name.trim())) {
        return dutyStatusMalfunctionTypes;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, DutyStatusMalfunctionTypes.class.getSimpleName(), DutyStatusMalfunctionTypes.NONE);
    return NONE;
  }
}
