/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.exceptionevent;

import static com.geotab.model.entity.exceptionevent.NoExceptionEvent.NO_EXCEPTION_EVENT_ID;

import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.entity.EntityWithVersion;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.diagnostic.Diagnostic;
import com.geotab.model.entity.exceptionevent.state.ExceptionEventState;
import com.geotab.model.entity.rule.Rule;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.DriverEmbeddedSerializer;
import com.geotab.util.Util;
import java.time.Duration;
import java.time.LocalDateTime;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * The event of an exception generated by {@link Rule} violation.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class ExceptionEvent extends EntityWithVersion {

  /**
   * The start date of the ExceptionEvent; at or after this date.
   */
  private LocalDateTime activeFrom;

  /**
   * The end date of the ExceptionEvent; at or before this date.
   */
  private LocalDateTime activeTo;

  /**
   * The {@link Device} related to the ExceptionEvent.
   */
  private Device device;

  /**
   * The distance travelled since the start of the ExceptionEvent.
   */
  private Float distance;

  /**
   * The duration of the exception event.
   */
  private Duration duration;

  /**
   * The exception {@link Rule} which was broken.
   */
  private Rule rule;

  /**
   * The {@link Diagnostic}.
   */
  private Diagnostic diagnostic;

  /**
   * The {@link User} driver specified for the device.
   */
  @JsonSerialize(using = DriverEmbeddedSerializer.class)
  private User driver;

  /**
   * The {@link ExceptionEventState} of the exception.
   */
  private ExceptionEventState state;

  /**
   * The time this exception event was last updated.
   */
  private LocalDateTime lastModifiedDateTime;

  public static ExceptionEvent fromSystem(String id) {
    if (Util.isEmpty(id)) return null;
    if (NO_EXCEPTION_EVENT_ID.equals(id)) return NoExceptionEvent.getInstance();
    return null;
  }
}
