/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.fuel;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.entity.Entity;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.user.Driver;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.DriverEmbeddedSerializer;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Fuel tax reporting element. The available driving history for a {@link Device} is stored as a sequence of such
 * details. Each next detail starts when and where the previous detail ended. A detail is identified by its parameters
 * (enter and exit time, odometer, GPS odometer, latitude and longitude) and its attributes (jurisdiction, {@link
 * Driver}, toll road, authority). When any of the attributes changes, the current detail ends and a new detail begins.
 * For more information, see <a href="https://docs.google.com/document/d/1vqQYJEIrUqOJ0LlFEeY1iVddcC-I4DTY2z73NE0Nzug">IFTA
 * Guide</a>.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class FuelTaxDetail extends Entity {

  /**
   * The {@link Device}.
   */
  private Device device;

  /**
   * The {@link Driver}.
   */
  @JsonSerialize(using = DriverEmbeddedSerializer.class)
  private User driver;

  /**
   * The jurisdiction, such as a U.S. state or a Canadian province.
   */
  private String jurisdiction;

  /**
   * The time at which the detail begins.
   */
  private LocalDateTime enterTime;

  /**
   * The odometer, in km, at the enter time.
   */
  private Double enterOdometer;

  /**
   * The GPS odometer, in km, at the enter time.
   */
  private Double enterGpsOdometer;

  /**
   * The latitude at the enter time.
   */
  private Double enterLatitude;

  /**
   * The longitude at the enter time.
   */
  private Double enterLongitude;

  /**
   * A value indicating whether the odometer at enter time is interpolated.
   */
  @JsonProperty("isEnterOdometerInterpolated")
  private Boolean isEnterOdometerInterpolated;

  /**
   * The time at which the detail ends.
   */
  private LocalDateTime exitTime;

  /**
   * The odometer, in km, at the exit time.
   */
  private Double exitOdometer;

  /**
   * The GPS odometer, in km, at the exit time.
   */
  private Double exitGpsOdometer;

  /**
   * The latitude at the exit time.
   */
  private Double exitLatitude;

  /**
   * The longitude at the exit time.
   */
  private Double exitLongitude;

  /**
   * A value indicating whether the odometer at exit time is interpolated.
   */
  @JsonProperty("isExitOdometerInterpolated")
  private Boolean isExitOdometerInterpolated;

  /**
   * The toll road name. null by default.
   */
  private String tollRoad;

  /**
   * The authority. 'None' by default.
   */
  private String authority;

  /**
   * The odometer values, in km, at each hour within the detail's time interval.
   */
  private List<Double> hourlyOdometer;

  /**
   * A list of values indicating whether the odometer at the corresponding hour is interpolated.
   */
  private List<Boolean> hourlyIsOdometerInterpolated;

  /**
   * The GPS odometer values, in km, at each hour within the detail's time interval.
   */
  private List<Double> hourlyGpsOdometer;

  /**
   * The latitude values at each hour within the detail's time interval.
   */
  private List<Double> hourlyLatitude;

  /**
   * The longitude values at each hour within the detail's time interval.
   */
  private List<Double> hourlyLongitude;

  /**
   * A value indicating whether the odometer values are based on engine odometer records and/or odometer adjustment
   * records, as opposed to GPS distances alone.
   */
  @JsonProperty("isClusterOdometer")
  private Boolean isClusterOdometer;

  /**
   * A value indicating whether the details has hourly data.
   */
  @JsonProperty("hasHourlyData")
  private Boolean hasHourlyData;

  /**
   * The last processed record versions.
   */
  private List<Long> versions;

  /**
   * A value indicating whether the detail is negligible by time or distance.
   */
  @JsonProperty("isNegligible")
  private Boolean isNegligible;

  /**
   * The attributes.
   */
  private FuelTaxAttributes fuelTaxAttributes;
}
