/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.fuel;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum FuelTransactionProductType implements HasName {

  /**
   * Unknown product type.
   */
  @JsonEnumDefaultValue
  UNKNOWN("Unknown", 0),

  /**
   * A non-fuel purchase.
   */
  NON_FUEL("NonFuel", 0x1),

  /**
   * Regular grade gasoline (86-87 Octane : 90-91 Ron).
   */
  REGULAR("Regular", 0x2),

  /**
   * Mid grade gasoline (88-89 Octane : 92-93 Ron).
   */
  MIDGRADE("Midgrade", 0x4),

  /**
   * Premium grade gasoline (90-91 Octane : 94-95 Ron).
   */
  PREMIUM("Premium", 0x8),

  /**
   * Super grade gasoline (92-94+ Octane : 96-99+ Ron).
   */
  SUPER("Super", 0x10),

  /**
   * Diesel fuel.
   */
  DIESEL("Diesel", 0x20),

  /**
   * E85 (Ethanol 85%).
   */
  E85("E85", 0x40),

  /**
   * CNG (Compressed Natural Gas).
   */
  CNG("CNG", 0x80),

  /**
   * LPG (Liquid Propane Gas).
   */
  LPG("LPG", 0x100),

  /**
   * Hydrogen.
   */
  HYDROGEN("Hydrogen", 0x200),

  /**
   * Diesel exhaust fluid.
   */
  DIESEL_EXHAUST_FLUID("DieselExhaustFluid", 0x400),

  /**
   * Electric.
   */
  ELECTRIC("Electric", 0x800);

  private final String name;
  private final int code;

  FuelTransactionProductType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public static FuelTransactionProductType findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          FuelTransactionProductType.class.getSimpleName(), FuelTransactionProductType.UNKNOWN);
      return UNKNOWN;
    }

    for (FuelTransactionProductType fuelTransactionProductType : values()) {
      if (fuelTransactionProductType.getName().equalsIgnoreCase(name.trim())) {
        return fuelTransactionProductType;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, FuelTransactionProductType.class.getSimpleName(), FuelTransactionProductType.UNKNOWN);
    return UNKNOWN;
  }
}
