/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.ioxaddon;

import com.geotab.model.entity.Entity;
import com.geotab.model.entity.device.Device;
import java.time.LocalDateTime;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Represents an Iox Add-On (like modem or navigation device) that is attached to a GO unit. Each Iox Add-On is assigned
 * a channel - which is the serial port number that it typically communicates with.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class IoxAddOn extends Entity {

  /**
   * The channel on which the Add-On is attached to the GO unit. This is typically just a serial port. 0 means the
   * Add-On is not attached.
   */
  private Integer channel;

  /**
   * The unique identifier for this Iox Add-On type. Iox Add-On types are assigned by Geotab. See {@link
   * KnownIoxAddOnType}.
   */
  private Integer type;

  /**
   * The {@link Device} this IoxAddOn is connected to.
   */
  private Device device;

  /**
   * The DateTime this IoxAddOn was assigned it's channel.
   */
  private LocalDateTime dateTime;

  public void setType(Integer type) {
    if (!isIoxAddOn(type)) throw new IllegalArgumentException("Invalid identifier " + type);
    this.type = type;
  }

  /**
   * Determines whether the provided identifier is for an Iox Add-On. See {@link KnownIoxAddOnType}.
   *
   * @param identifier The identifier.
   * @return <c>true</c> if the identifier is for a Iox Add-On; otherwise, <c>false</c>.
   */
  public static boolean isIoxAddOn(Integer identifier) {
    return identifier != null && identifier >= 4096 && identifier < 8192;
  }
}
