/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.logrecord;

import static com.geotab.util.LogRecordValidator.isValidCoordinateRange;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.geotab.model.coordinate.Coordinate;
import com.geotab.model.entity.Entity;
import com.geotab.model.entity.device.Device;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Record of log entries containing data for a device's position and speed at a specific date and time.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class LogRecord extends Entity {

  private Double latitude;
  private Double longitude;
  private Float speed;
  private Device device;

  private LocalDateTime dateTime;

  /**
   * Calculates the distance between two lat/long points.
   *
   * @param longitudeA The longitude a.
   * @param latitudeA  The latitude a.
   * @param longitudeB The longitude b.
   * @param latitudeB  The latitude b.
   * @return The distance.
   */
  public static double distanceBetweenPoints(double longitudeA, double latitudeA, double longitudeB, double latitudeB) {
    return Coordinate.distanceBetween(longitudeA, latitudeA, longitudeB, latitudeB) / 1000;
  }

  /**
   * Calculates {@link Coordinate} list representing an extent that fully covers the {@link LogRecord}(s).
   *
   * @param logRecords The log records.
   * @return The {@link Coordinate} list.
   */
  @SuppressWarnings("LocalVariableName")
  public static List<Coordinate> getExtent(List<LogRecord> logRecords) {
    double xMin = Double.MAX_VALUE;
    double xMax = Double.MIN_VALUE;
    double yMin = Double.MAX_VALUE;
    double yMax = Double.MIN_VALUE;
    for (LogRecord log : logRecords) {
      double x = log.longitude;
      double y = log.latitude;
      if (xMin != Double.MAX_VALUE) {
        if (xMin > x) {
          if (!isValidCoordinateRange(x, y)) {
            continue;
          }
          xMin = x;
        } else if (xMax < x) {
          if (!isValidCoordinateRange(x, y)) {
            continue;
          }
          xMax = x;
        }
        if (yMin > y) {
          if (isValidCoordinateRange(x, y)) {
            yMin = y;
          }
        } else if (yMax < y && isValidCoordinateRange(x, y)) {
          yMax = y;
        }
      } else if (isValidCoordinateRange(x, y)) {
        xMax = xMin = x;
        yMax = yMin = y;
      }
    }
    if (xMin == Double.MAX_VALUE) {
      return null;
    }
    return Coordinate.rectangleFromLtrb(xMin, yMax, xMax, yMin);
  }

  /**
   * Calculates the distance between two lat/long points in kilometers.
   *
   * @param logRecord {@link LogRecord}
   * @return Result in kilometers.
   */
  @JsonIgnore
  public double distanceTo(LogRecord logRecord) {
    return distanceBetweenPoints(longitude, latitude, logRecord.longitude, logRecord.latitude);
  }

  /**
   * Calculates the distance between two lat/long points in kilometers.
   *
   * @param coordinate {@link Coordinate}
   * @return Result in kilometers.
   */
  @JsonIgnore
  public double distanceTo(Coordinate coordinate) {
    return distanceBetweenPoints(longitude, latitude, coordinate.getX(), coordinate.getY());
  }

  /**
   * Gets the {@link Coordinate} of the log record.
   *
   * @return The coordinate of the log.
   */
  @JsonIgnore
  public Coordinate toSimpleCoordinate() {
    if (longitude == null || latitude == null) {
      return null;
    }
    return new Coordinate(longitude, latitude);
  }
}
