/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.recipient;

import static com.geotab.util.Util.isEmpty;

import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.entity.Entity;
import com.geotab.model.entity.group.Group;
import com.geotab.model.entity.notification.NotificationBinaryFile;
import com.geotab.model.entity.rule.Rule;
import com.geotab.model.entity.trip.TripType;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.EntityAsIdSerializer;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * The recipient for a specific notification. A recipient is linked to {@link Rule}(s) via a DistributionList. When a
 * {@link Rule} is violated the DistributionList linked recipient receives a notification. The type of recipient is
 * defined by it's {@link RecipientType}. Not all properties of this object will have a value at the same time they are
 * dependent on the {@link RecipientType}. Recipient is represented by the string {@link NoRecipient#NO_RECIPIENT_ID}
 * where there is no recipient.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class Recipient extends Entity {

  /**
   * The email address used when sending notifications via Email.
   */
  private String address;

  /**
   * The {@link TripType} to assign the related device to.
   */
  private TripType tripType;

  /**
   * The {@link Group} to assign the related device to.
   */
  private Group group;

  /**
   * The {@link NotificationBinaryFile} to notify with.
   */
  private NotificationBinaryFile notificationBinaryFile;

  /**
   * The {@link RecipientType} (type of notification message) this instance refers to.
   */
  private RecipientType recipientType;

  /**
   * The {@link User} to receive notification.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User user;

  public static Recipient fromSystem(String id) {
    if (isEmpty(id)) return null;
    if (NoRecipient.NO_RECIPIENT_ID.equalsIgnoreCase(id)) return NoRecipient.getInstance();
    return null;
  }
}
