/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.recipient;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.entity.device.GoDevice;
import com.geotab.model.entity.group.Group;
import com.geotab.model.entity.notification.NotificationBinaryFile;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.HasName;

public enum RecipientType implements HasName {

  /**
   * Fallback when enumeration value is unknown.
   */
  @JsonEnumDefaultValue
  UNKNOWN("Unknown", -1),

  /**
   * Recipient will be notified via a normal priority log that will appear in their Notifications. A {@link User} is
   * required.
   */
  LOG_ONLY("LogOnly", 0),

  /**
   * Recipient will be notified via either a simple email. A {@link User} or valid Address is required. Optionally
   * includes an EmailTemplate ({@link NotificationBinaryFile}).
   */
  EMAIL("Email", 1),

  /**
   * Recipient will be notified with a popup notification in the MyGeotab application that will also appear in their
   * Notifications. A {@link User} is required.
   */
  LOG_POPUP("LogPopup", 2),

  /**
   * Recipient will be notified with an urgent popup in the MyGeotab application that will also appear in their
   * Notifications. A {@link User} is required.
   */
  LOG_URGENT_POPUP("LogUrgentPopup", 3),

  /**
   * Recipient will be notified with an alarm sound in the MyGeotab application. A {@link User} and {@link
   * NotificationBinaryFile} are required.
   */
  ALARM("Alarm", 4),

  /**
   * Recipient will be notified via the related Address.
   */
  WEB_SERVICE("WebService", 5),

  /**
   * Recipient will be notified via a text message in the vehicle (for example, Garmin).
   */
  TEXT_MESSAGE("TextMessage", 6),

  /**
   * Recipient must be prompted via a text message in the vehicle (for example, Garmin) to select their status.
   */
  CHANGE_STATUS("ChangeStatus", 7),

  /**
   * Beep three times rapidly using the GO device buzzer.
   */
  BEEP_THREE_TIMES_RAPIDLY("BeepThreeTimesRapidly", 8),

  /**
   * Beep three times using the GO device buzzer.
   */
  BEEP_THREE_TIMES("BeepThreeTimes", 9),

  /**
   * Beep ten times rapidly using the GO device buzzer.
   */
  BEEP_TEN_TIMES_RAPIDLY("BeepTenTimesRapidly", 10),

  /**
   * Recipient will be notified with a popup notification in the MyGeotab application that will also appear in their
   * Notifications.
   */
  LOG_POPUP_INFO("LogPopupInfo", 11),

  /**
   * Recipient will be notified via the related WebRequestTemplate (NotificationBinaryFile). A WebRequestTemplate is
   * required.
   */
  WEB_REQUEST("WebRequest", 12),

  /**
   * The vehicle associate with the ExceptionEvent will be assigned to the designated group (removed from sibling
   * groups). A {@link Group} is required.
   */
  ASSIGN_TO_GROUP("AssignToGroup", 13),

  /**
   * Recipient will be notified via audio from a text to speech application connected to the {@link GoDevice}.
   */
  TEXT_TO_SPEECH("TextToSpeech", 14),

  /**
   * Recipient will be notified with a popup notification in the MyGeotab application that will also appear in their
   * Notifications. A {@link User} is required.
   */
  LOG_NEWS_POPUP("LogNewsPopup", 15),

  /**
   * Beep three times rapidly using the GO device buzzer. The device will beep even if there is a delay in the data and
   * may beep after the event causing the exception has ended.
   */
  BEEP_THREE_TIMES_RAPIDLY_ALLOW_DELAY("BeepThreeTimesRapidlyAllowDelay", 16),

  /**
   * Beep three times using the GO device buzzer. The device will beep even if there is a delay in the data and may beep
   * after the event causing the exception has ended.
   */
  BEEP_THREE_TIMES_ALLOW_DELAY("BeepThreeTimesAllowDelay", 17),

  /**
   * Beep ten times rapidly using the GO device buzzer. The device will beep even if there is a delay in the data and
   * may beep after the event causing the exception has ended.
   */
  BEEP_TEN_TIMES_RAPIDLY_ALLOW_DELAY("BeepTenTimesRapidlyAllowDelay", 18),

  /**
   * Recipient will be notified via audio from a GoTalk IOX. The GoTalk will speak even if there is a delay in the data
   * and may beep after the event causing the exception has ended.
   */
  TEXT_TO_SPEECH_ALLOW_DELAY("TextToSpeechAllowDelay", 19),

  /**
   * TripType change will be added.
   */
  TRIP_TYPE_CHANGE("TripTypeChange", 20),

  /**
   * Recipients within a group will be notified via simple email. A valid group name is required. Optionally includes an
   * EmailTemplate (NotificationBinaryFile).
   */
  EMAIL_TO_GROUP("EmailToGroup", 21),

  /**
   * Hos will be enabled.
   */
  HOS_ENABLED("HosEnabled", 22),

  /**
   * Hos will be disabled.
   */
  HOS_DISABLED("HosDisabled", 23),

  /**
   * DeviceDataPrivacy change will be added.
   */
  DEVICE_DATA_PRIVACY("DeviceDataPrivacy", 24),

  /**
   * ShareableLink change will be expired.
   */
  EXPIRE_SHAREABLE_LINK("ExpireShareableLink", 25);

  private final String name;
  private final int code;

  RecipientType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }
}
