/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.rule;

import static com.geotab.model.entity.rule.NoRule.NO_RULE_ID;

import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.Id;
import com.geotab.model.drawing.Color;
import com.geotab.model.entity.NameEntityWithVersion;
import com.geotab.model.entity.condition.Condition;
import com.geotab.model.entity.group.Group;
import com.geotab.model.serialization.EntityCollectionAsIdCollectionSerializer;
import com.geotab.util.Util;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * A rule is the definition of conditions that, when "violated", will generate an ExceptionEvent.
 *
 * <p>The rule's logic is defined by it's tree of {@link Condition}(s).
 *
 * <p>It's condition tree will be evaluated against data for device(s) that are members of the
 * rule's assigned group(s) or the device(s)/driver(s) defined in the rule condition tree. The conditions will be
 * evaluated independently against the assets in the selected groups.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class Rule extends NameEntityWithVersion {

  /**
   * The hierarchical tree of {@link Condition}(s) defining the logic of a rule. A rule should have one or more
   * conditions in it's tree.
   */
  private Condition condition;

  /**
   * Start date of the Rule's notification activity period.
   *
   * <p>The events with earlier date than this date will not be reported through the notification
   * engine.
   */
  private LocalDateTime activeFrom;

  /**
   * End date of the Rule's notification activity period.
   */
  private LocalDateTime activeTo;

  /**
   * The {@link ExceptionRuleBaseType} of the rule; either Custom, Stock or ZoneStop.
   */
  private ExceptionRuleBaseType baseType;

  /**
   * The {@link Color} associated with this rule. Used when rendering ExceptionEvent(s) related to this rule. Color is
   * defined by the parameters "Red", "Green" and "Blue".
   */
  private Color color;

  /**
   * Free text field where any user information can be stored and referenced for this entity.
   */
  private String comment;

  /**
   * A list of {@link Group}(s) assigned to the rule. Device in these groups will have the rule evaluated against their
   * data.
   *
   * <p>Device conditions will override devices in the assigned groups.
   */
  @JsonSerialize(converter = EntityCollectionAsIdCollectionSerializer.class)
  private List<Group> groups;

  public Rule(String id) {
    setId(new Id(id));
    setName(id);
  }

  public static Rule fromSystem(String id) {
    if (Util.isEmpty(id)) return null;
    if (NO_RULE_ID.equals(id)) return NoRule.getInstance();
    return null;
  }
}
