/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.textmessage;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.entity.EntityWithVersion;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.EntityAsIdSerializer;
import com.geotab.model.serialization.TextMessageDeserializer;
import java.time.LocalDateTime;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * A message to send or received from a device.
 *
 * <p>When working with text messages it is important to make the distinction between a "Reply"
 * and a "Response".
 *
 * <ul>
 *     <li>
 *         Reply: A reply is a Text Message that is a Reply to another text message.
 *         For example: A text message is sent to a device and the device replies with
 *         a text message of it's own.
 *     </li>
 *     <li>s
 *         Response: A response is a predefined ("canned") response within a text message.
 *         For example: A text message is sent to a device with a number of canned responses to
 *         reply with (Yes, No, Maybe).
 *         One of those responses is selected by the driver and is the message of the Reply
 *         from the device.
 *     </li>
 * </ul>
 */
@Data
@NoArgsConstructor
@SuperBuilder
@JsonDeserialize(using = TextMessageDeserializer.class)
public class TextMessage extends EntityWithVersion {

  /**
   * User that read the message.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User markedReadBy;

  /**
   * The user that sent the message or null.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User user;

  /**
   * The user the message was sent to.  or null.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User recipient;

  /**
   * Date message was sent.
   */
  private LocalDateTime sent;

  /**
   * Date message was delivered.
   */
  private LocalDateTime delivered;

  /**
   * The device that send/received the message.
   */
  private Device device;

  /**
   * The message content. A basic message can be sent via a TextContent or a message with predefined ("canned")
   * responses by a CannedResponseContent or a series of LocationContent comprising a route.
   */
  private TextMessageContentType messageContent;

  /**
   * Message is to or from a vehicle.
   */
  @JsonProperty("isDirectionToVehicle")
  private Boolean isDirectionToVehicle;

  /**
   * The date and time the message was read or null if the message has not been read.
   */
  private LocalDateTime read;

  /**
   * The text message that this is the reply to. Only a reply message will have a parent.
   */
  private TextMessage parentMessage;

  /**
   * The text message reply to this message. A text message should only have one reply. (Should not reply to a reply).
   */
  private TextMessage replyMessage;

  /**
   * The date message is active from.
   */
  private LocalDateTime activeFrom;

  /**
   * The date message is active to.
   */
  private LocalDateTime activeTo;

  /**
   * The message length.
   */
  private Integer messageSize;
}
