/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.zone;

import static com.geotab.model.entity.zone.NoZone.NO_ZONE_ID;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.Id;
import com.geotab.model.coordinate.Coordinate;
import com.geotab.model.drawing.Color;
import com.geotab.model.entity.NameEntityWithVersion;
import com.geotab.model.entity.group.Group;
import com.geotab.model.entity.zone.type.ZoneType;
import com.geotab.model.serialization.EntityCollectionAsIdCollectionSerializer;
import com.geotab.util.Util;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Sometimes referred to as a "Geofence", a zone is a virtual geographic boundary, defined by its points representing a
 * real-world geographic area.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class Zone extends NameEntityWithVersion {

  /**
   * A free text field where any user information can be stored and referenced for this entity. Default [""].
   */
  private String comment;

  /**
   * A value indicating whether this zone must be displayed when viewing a map or it should be hidden. Default [true].
   */
  private Boolean displayed;

  /**
   * The list of {@link ZoneType}(s) this zone belongs to. Default [Customer].
   */
  private List<ZoneType> zoneTypes;

  /**
   * The list of points (see {@link Coordinate}) that make up this zone. A zone should be closed, the first point is the
   * same coordinate as the last point.
   *
   * <p>It is possible to add a zone with an unclosed set of points but the points will be closed
   * by the system.
   */
  private List<Coordinate> points;

  /**
   * The date indicating when this zone begins it's active lifespan. Default [UtcNow].
   */
  private LocalDateTime activeFrom;

  /**
   * The date indicating when this zone finishes it's active lifespan. Default [MaxDate].
   */
  private LocalDateTime activeTo;

  /**
   * The {@link Color} of the fill for this zone when showing on a map. Default [based on zone type; Customer: Orange,
   * Office: Light Orange, Home: Green, Other: Blue].
   */
  private Color fillColor;

  /**
   * Whether this zone name must be shown when devices stop in this zone. If [true] a "zone stop rule" (Rule with
   * BaseType: ZoneStop) will automatically be created for this zone. This is to facilitate reporting on zone stops. The
   * rule is not visible via the UI. Default [true].
   */
  private Boolean mustIdentifyStops;

  /**
   * The {@link Group}(s) this zone belongs to.
   */
  @JsonSerialize(converter = EntityCollectionAsIdCollectionSerializer.class)
  private List<Group> groups;

  /**
   * The external reference. Any type of external reference you would like to attach to the zone. For example; an ID
   * from another data source referenced when exporting zone data into another program. Maximum length [255] Default
   * [""].
   */
  private String externalReference;

  @JsonIgnore
  @Override
  public Long getVersion() {
    return super.getVersion();
  }

  public Zone(String id) {
    setId(new Id(id));
    setName(id);
  }

  public static Zone fromSystem(String id) {
    if (Util.isEmpty(id)) return null;
    if (NO_ZONE_ID.equals(id)) return NoZone.getInstance();
    return null;
  }
}
