/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.error;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.serialization.HasName;

public enum JsonRpcErrorCode implements HasName {

  /**
   * Fallback when enumeration value is unknown.
   */
  @JsonEnumDefaultValue
  UNKNOWN("Unknown", -1),

  /**
   * Invalid JSON was received by the server. An error occurred on the server while parsing the JSON text.
   */
  PARSE_ERROR("ParseError", -32700),

  /**
   * Internal JSON-RPC error.
   */
  INTERNAL_ERROR("InternalError", -32603),

  /**
   * Invalid method parameter(s).
   */
  INVALID_PARAMS("InvalidParams", -32602),

  /**
   * The method does not exist / is not available.
   */
  METHOD_NOT_FOUND("MethodNotFound", -32601),

  /**
   * The JSON sent is not a valid Request object.
   */
  INVALID_REQUEST("InvalidRequest", -32600),

  /**
   * The server responded with GroupRelationViolatedException.
   */
  GROUP_RELATION_VIOLATED_ERROR("GroupRelationViolatedError", -32599),

  /**
   * The server responded with DbUnavailableException, State= DbUnavailableState.None.
   */
  DB_UNAVAILABLE_GENERAL_ERROR("DbUnavailableGeneralError", -32590),

  /**
   * The server responded with DbUnavailableException, State= DbUnavailableState.ConnectionFailure.
   */
  DB_UNAVAILABLE_CONNECTION_FAILURE("DbUnavailableConnectionFailure", -32589),

  /**
   * The server responded with DbUnavailableException, State= DbUnavailableState.UnknownDatabase.
   */
  DB_UNAVAILABLE_UNKNOWN_DATABASE("DbUnavailableUnknownDatabase", -32588),

  /**
   * The server responded with DbUnavailableException, State= DbUnavailableState.Initializing.
   */
  DB_UNAVAILABLE_INITIALIZING("DbUnavailableInitializing", -32587),

  /**
   * The server responded with DbUnavailableException, State= DbUnavailableState.OperationAborted.
   */
  DB_UNAVAILABLE_OPERATION_ABORTED("DbUnavailableOperationAborted", -32586),

  /**
   * Invalid certificate error.
   */
  INVALID_CERTIFICATE_EXCEPTION("InvalidCertificateException", -32001),

  /**
   * Internal server error.
   */
  INTERNAL_SERVER_ERROR("InternalServerError", -32000),

  /**
   * The default value.
   */
  NONE("None", 0);

  private final String name;
  private final int code;

  JsonRpcErrorCode(String name, int code) {
    this.name = name;
    this.code = code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public int getCode() {
    return code;
  }

  public static JsonRpcErrorCode getByCode(Integer code) {
    if (code == null) {
      return null;
    }
    for (JsonRpcErrorCode value : values()) {
      if (value.getCode() == code) {
        return value;
      }
    }
    return null;
  }
}
