/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.exceptionevent.ExceptionEvent;
import java.time.LocalDateTime;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link ExceptionEvent}.
 *
 * <p>This search has been designed to work efficiently with these combinations of parameters:
 * <ul>
 *      <li>
 *             Id
 *     </li>
 *      <li>
 *             DeviceSearch + RuleSearch + FromDate and/or ToDate
 *     </li>
 * </ul>
 */
@Data
@NoArgsConstructor
public class ExceptionEventSearch extends IdSearch {

  /**
   * Search for Exception Events that occurred at this date or after.
   */
  private LocalDateTime fromDate;

  /**
   * Search for Exception Events that occurred at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * Filter by the {@link DeviceSearch} options. Providing a Device ID will search for any Exception Events recorded for
   * that Device. Providing Groups will search Exception Events recorded for Devices that are members of the provided
   * GroupSearch(s) or their children.
   *
   * <p>Available DeviceSearch options are:
   * <ul>
   *      <li>
   *             Id
   *     </li>
   *      <li>
   *             Groups
   *     </li>
   * </ul>
   */
  private DeviceSearch deviceSearch;

  /**
   * Filter by the {@link RuleSearch} options. Providing a Rule ID will search for any Exception Events recorded for
   * that Rule.
   *
   * <p>Available RuleSearch options are:
   * <ul>
   *      <li>
   *             Id
   *     </li>
   * </ul>
   */
  private RuleSearch ruleSearch;

  @Builder
  public ExceptionEventSearch(String id, LocalDateTime fromDate, LocalDateTime toDate,
      DeviceSearch deviceSearch, RuleSearch ruleSearch) {
    super(id);
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.deviceSearch = deviceSearch;
    this.ruleSearch = ruleSearch;
  }
}
