/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.fuel.FuelTaxDetail;
import java.time.LocalDateTime;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link FuelTaxDetail} elements.
 *
 * <p>This search has been designed to work efficiently with these parameters:
 * <ul>
 *    <li>deviceSearch</li>
 *    <li>fromDate</li>
 *    <li>toDate</li>
 * </ul>
 */
@Data
@NoArgsConstructor
public class FuelTaxDetailSearch extends IdSearch {

  /**
   * Filter by the {@link DeviceSearch} options. Providing a device ID will search for any fuel tax details that are
   * assigned to that Device. Providing the Groups will search for fuel tax details for that have Devices in that
   * group.
   * <ul>
   *    <li>id</li>
   *    <li>groups</li>
   * </ul>
   */
  private DeviceSearch deviceSearch;

  /**
   * The beginning of the time interval. The search will adjust it to the nearest hour on or before this time. For
   * instance, 8:20 AM will become 8 AM.
   */
  private LocalDateTime fromDate;

  /**
   * The end of the time interval. The search will adjust it to the nearest hour on or after this time. For instance,
   * 5:40 PM will become 6 PM.
   */
  private LocalDateTime toDate;

  /**
   * A value indicating whether to include hourly data.
   */
  private Boolean includeHourlyData;

  /**
   * A value indicating whether to include any parts of boundary details that fall outside the time interval.
   */
  private Boolean includeBoundaries;

  @Builder
  public FuelTaxDetailSearch(String id, DeviceSearch deviceSearch, LocalDateTime fromDate,
      LocalDateTime toDate, Boolean includeHourlyData, Boolean includeBoundaries) {
    super(id);
    this.deviceSearch = deviceSearch;
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.includeHourlyData = includeHourlyData;
    this.includeBoundaries = includeBoundaries;
  }
}
