/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import static java.lang.Boolean.TRUE;

import com.geotab.model.entity.fuel.FuelTransaction;
import com.geotab.model.entity.fuel.FuelTransactionProvider;
import java.time.LocalDateTime;
import java.util.Optional;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The criteria used when searching for a {@link FuelTransaction}.
 */
@Data
@NoArgsConstructor
public class FuelTransactionSearch extends IdSearch {

  /**
   * The object used to specify the arguments when searching for a {@link FuelTransaction}.
   */
  private LocalDateTime fromDate;

  /**
   * The to date, which is used to search for {@link FuelTransaction}s that occur on or before this date.
   */
  private LocalDateTime toDate;

  /**
   * The maximum version for which {@link FuelTransaction}s should be searched.
   */
  private Long maxVersion;

  /**
   * The VehicleIdentificationNumber.
   */
  private String vehicleIdentificationNumber;

  /**
   * A value indicating whether to include source data with transaction. Source data can be a large string, therefore,
   * optionally exclude it from results.
   */
  private Boolean includeSourceData;

  /**
   * Search for Fuel Transactions with this External Reference. Wildcard can be used by prepending/appending "%" to
   * string. Example "%reference%".
   */
  private String externalReference;

  /**
   * Search by the {@link FuelTransactionProvider}.
   */
  private FuelTransactionProvider provider;

  /**
   * Search the {@link FuelTransaction}s for this {@link DeviceSearch}.
   *
   * <p>Available DeviceSearch options are:.
   * <ul>
   *    <li>id</li>
   * </ul>
   * <ul>
   *    <li>groups</li>
   * </ul>
   */
  private DeviceSearch deviceSearch;

  @Builder
  public FuelTransactionSearch(String id, LocalDateTime fromDate, LocalDateTime toDate,
      Long maxVersion, String vehicleIdentificationNumber, Boolean includeSourceData,
      String externalReference, FuelTransactionProvider provider,
      DeviceSearch deviceSearch) {
    super(id);
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.maxVersion = maxVersion;
    this.vehicleIdentificationNumber = vehicleIdentificationNumber;
    this.includeSourceData = Optional.ofNullable(includeSourceData).orElse(TRUE);
    this.externalReference = externalReference;
    this.provider = provider;
    this.deviceSearch = deviceSearch;
  }
}
