/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.Id;
import com.geotab.model.entity.logrecord.LogRecord;
import java.time.LocalDateTime;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for  {@link LogRecord}.
 *
 * <p>When searching for log records the system will return all records that match the search
 * criteria and interpolate the value at the provided from/to dates when there is no record that corresponds to the
 * date. Interpolated records are dynamically created when the request is made and can be identified as not having the
 * ID property populated. Records with an ID are stored in the database.
 *
 * <p>This search has been designed to work efficiently with these combinations of parameters:
 * <ul>
 *    <li>
 *      {@link Id}
 *    </li>
 *    <li>
 *       deviceSearch + fromDate and/or toDate
 *    </li>
 * </ul>
 */
@Data
@NoArgsConstructor
public class LogRecordSearch extends IdSearch {

  /**
   * Gets or sets search for LogRecords at this date or after.
   */
  private LocalDateTime fromDate;

  /**
   * Gets or sets search for LogRecords at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * Gets or sets search for LogRecords for this device id.
   *
   * <p>Available DeviceSearch options are:
   *  <ul>
   *     <li>
   *         {@link Id}
   *     </li>
   * </ul>
   */
  private Id deviceSearch;

  @Builder
  public LogRecordSearch(String id, LocalDateTime fromDate, LocalDateTime toDate,
      Id deviceSearch) {
    super(id);
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.deviceSearch = deviceSearch;
  }
}
