/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.statusdata.StatusData;
import java.time.LocalDateTime;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link StatusData}.
 *
 * <p>When searching for status data including DeviceSearch and DiagnosticSearch the system will
 * return all records that match the search criteria and interpolate the value at the provided from/to dates when there
 * is no record that corresponds to the date. Interpolated records are dynamically created when the request is made and
 * can be identified as not having the ID property populated. Records with an ID are stored in the database.
 *
 * <p>This search has been designed to work efficiently with these combinations of parameters:
 * <ul>
 *      <li>
 *             Id
 *     </li>
 *      <li>
 *             DeviceSearch + DiagnosticSearch + FromDate and/or ToDate
 *     </li>
 * </ul>
 *
 * <p>Cannot be used with DriverGroups.
 */
@Data
@NoArgsConstructor
public class StatusDataSearch extends IdSearch {

  /**
   * Search for {@link StatusData} recorded for this DeviceSearch Id
   *
   * <p>Available DeviceSearch options are:.
   * <ul>
   *    <li>id</li>
   * </ul>
   */
  private DeviceSearch deviceSearch;

  /**
   * Search for {@link StatusData} with this DiagnosticSearch Id
   *
   * <p>Available DiagnosticSearch options are:.
   * <ul>
   *    <li>id</li>
   * </ul>
   */
  private DiagnosticSearch diagnosticSearch;

  /**
   * Search for {@link StatusData} records that were logged at this date or after.
   */
  private LocalDateTime fromDate;

  /**
   * Search for {@link StatusData} records that were logged at this date or before.
   */
  private LocalDateTime toDate;

  @Builder
  public StatusDataSearch(String id, DeviceSearch deviceSearch,
      DiagnosticSearch diagnosticSearch, LocalDateTime fromDate, LocalDateTime toDate) {
    super(id);
    this.deviceSearch = deviceSearch;
    this.diagnosticSearch = diagnosticSearch;
    this.fromDate = fromDate;
    this.toDate = toDate;
  }
}
