/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.serialization;

import static com.geotab.model.serialization.filter.FaultDataFilterProvider.FAULT_DATA_FILTER;
import static com.geotab.util.Util.isEmpty;
import static java.math.BigDecimal.ONE;
import static java.math.BigDecimal.ZERO;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.BeanDescription;
import com.fasterxml.jackson.databind.DeserializationConfig;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.KeyDeserializer;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.deser.BeanDeserializerModifier;
import com.fasterxml.jackson.databind.deser.ResolvableDeserializer;
import com.fasterxml.jackson.databind.deser.std.StdKeyDeserializer;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.impl.SimpleFilterProvider;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.geotab.model.Id;
import com.geotab.model.entity.Entity;
import com.geotab.model.entity.controller.Controller;
import com.geotab.model.entity.enginetype.EngineType;
import com.geotab.model.entity.exceptionevent.ExceptionEvent;
import com.geotab.model.entity.exceptionevent.state.ExceptionEventState;
import com.geotab.model.entity.failuremode.FailureMode;
import com.geotab.model.entity.faultdata.FaultStatus;
import com.geotab.model.entity.notification.NotificationBinaryFile;
import com.geotab.model.entity.parametergroup.ParameterGroup;
import com.geotab.model.entity.recipient.Recipient;
import com.geotab.model.entity.rule.Rule;
import com.geotab.model.entity.source.Source;
import com.geotab.model.entity.unitofmeasure.UnitOfMeasure;
import com.geotab.model.entity.worktime.WorkTime;
import com.geotab.model.entity.zone.Zone;
import com.geotab.model.entity.zone.type.ZoneType;
import com.geotab.model.serialization.filter.FaultDataFilterProvider;
import com.geotab.util.Util.FailableFunction;
import java.io.IOException;
import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 * Json serializer singleton which configures and keeps a single reference of the {@link ObjectMapper} to be used by the
 * {@link com.geotab.api.GeotabApi}.
 */
public final class ApiJsonSerializer {

  private static class InstanceHolder {

    private static final ApiJsonSerializer INSTANCE = new ApiJsonSerializer();
  }

  private ObjectMapper objectMapper;

  private ApiJsonSerializer() {
    this.objectMapper = buildDefaultObjectMapper();
  }

  public static ApiJsonSerializer getInstance() {
    return InstanceHolder.INSTANCE;
  }

  /**
   * Build default configuration for {@link ObjectMapper}.
   *
   * @return The {@link ObjectMapper}.
   */
  public ObjectMapper buildDefaultObjectMapper() {
    return JsonMapper.builder()
        .enable(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_USING_DEFAULT_VALUE)
        .enable(MapperFeature.ACCEPT_CASE_INSENSITIVE_ENUMS)
        .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
        .serializationInclusion(Include.NON_NULL)
        .addModule(new JavaTimeModule()) // added here so it has LESS priority
        .addModule(new SimpleModule("api-long-serialization-module")
            .addSerializer(long.class, new LongSerializer())
            .addSerializer(Long.class, new LongSerializer())
            .addDeserializer(long.class, new LongDeserializer())
            .addDeserializer(Long.class, new LongDeserializer()))
        .addModule(new SimpleModule("api-double-serialization-module")
            .addSerializer(double.class, new DoubleSerializer())
            .addSerializer(Double.class, new DoubleSerializer())
            .addSerializer(double[].class, new DoublePrimitiveArraySerializer())
            .addSerializer(Double[].class, new DoubleArraySerializer()))
        .addModule(new SimpleModule("api-float-serialization-module")
            .addSerializer(float.class, new FloatSerializer())
            .addSerializer(Float.class, new FloatSerializer())
            .addSerializer(float[].class, new FloatPrimitiveArraySerializer())
            .addSerializer(Float[].class, new FloatArraySerializer()))
        .addModule(new SimpleModule("api-time-serialization-module")
            .addSerializer(LocalDateTime.class, new LocalDateTimeSerializer())
            .addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer())
            .addSerializer(Duration.class, new DurationSerializer())
            .addDeserializer(Duration.class, new DurationDeserializer()))
        .addModule(new SimpleModule("api-custom-serialization-module")
            .setSerializerModifier(new ApiCustomSerializerModifier())
            .setDeserializerModifier(new ApiCustomDeserializerModifier()))
        .addModule(new SimpleModule("api-system-deserializers-module")
            .setDeserializerModifier(new MyBeanDeserializerModifier()))
        .filterProvider(new SimpleFilterProvider()
            .addFilter(FAULT_DATA_FILTER, new FaultDataFilterProvider()))
        .build();
  }

  /**
   * <b>!!! Use with caution !!!</b>
   *
   * <p>This method represents an extension point for the json serialization. It's recommended that
   * the API uses the default serialization config, but in case custom serialization config is needed, then this method
   * can be used.
   *
   * <p>Recommendation is to start from {@link #buildDefaultObjectMapper()}, add
   * custom config and then call this method before instantiating {@link com.geotab.api.GeotabApi}.
   *
   * @param objectMapper The custom {@link ObjectMapper}.
   */
  public void withCustomObjectMapper(ObjectMapper objectMapper) {
    if (objectMapper == null) {
      throw new IllegalArgumentException("ObjectMapper can not be null !");
    }
    this.objectMapper = objectMapper;
  }

  /**
   * Get the configured {@link ObjectMapper}.
   *
   * @return The {@link ObjectMapper}.
   */
  public ObjectMapper getObjectMapper() {
    return objectMapper;
  }

  static <T extends Number> void serializeNumberWithDecimals(T number, JsonGenerator jg) throws IOException {
    BigDecimal i = BigDecimal.valueOf(number.doubleValue()).stripTrailingZeros();
    if (/*is long*/i.signum() == 0 || i.scale() <= 0 || i.remainder(ONE).compareTo(ZERO) == 0) {
      jg.writeNumber(i.longValue());
    } else if (number instanceof Float) {
      jg.writeNumber(i.floatValue());
    } else {
      jg.writeNumber(i.doubleValue());
    }
  }

  static class MyBeanDeserializerModifier extends BeanDeserializerModifier {

    @Override
    public JsonDeserializer<?> modifyDeserializer(DeserializationConfig c, BeanDescription b, JsonDeserializer<?> d) {
      Class<?> t = b.getBeanClass();//@formatter:off
      if (Recipient.class.isAssignableFrom(t)) return sys(d, Recipient::fromSystem, id -> Recipient.builder().id(id).build());
      if (Controller.class.isAssignableFrom(t)) return sys(d, Controller::fromSystem, id -> Controller.builder().id(id).build());
      if (Rule.class.isAssignableFrom(t)) return sys(d, Rule::fromSystem, Rule::new);
      if (Source.class.isAssignableFrom(t)) return sys(d, Source::fromSystem, Source::new);
      if (EngineType.class.isAssignableFrom(t)) return sys(d, EngineType::fromSystem, id -> EngineType.builder().id(id).build());
      if (ExceptionEvent.class.isAssignableFrom(t)) return sys(d, ExceptionEvent::fromSystem, id -> ExceptionEvent.builder().id(id).build());
      if (ExceptionEventState.class.isAssignableFrom(t)) return sys(d, ExceptionEventState::fromSystem, id -> ExceptionEventState.builder().id(id).build());
      if (FailureMode.class.isAssignableFrom(t)) return sys(d, FailureMode::fromSystem, id -> FailureMode.builder().id(id).build());
      if (FaultStatus.class.isAssignableFrom(t)) return sys(d, FaultStatus::fromSystem, FaultStatus::new);
      if (NotificationBinaryFile.class.isAssignableFrom(t)) return sys(d, NotificationBinaryFile::fromSystem, NotificationBinaryFile::new);
      if (ParameterGroup.class.isAssignableFrom(t)) return sys(d, ParameterGroup::fromSystem, ParameterGroup::new);
      if (ZoneType.class.isAssignableFrom(t)) return sys(d, ZoneType::fromSystem, ZoneType::new);
      if (Zone.class.isAssignableFrom(t)) return sys(d, Zone::fromSystem, Zone::new);
      if (WorkTime.class.isAssignableFrom(t)) return sys(d, WorkTime::fromSystem, WorkTime::new);
      if (UnitOfMeasure.class.isAssignableFrom(t)) return sys(d, UnitOfMeasure::fromSystem, UnitOfMeasure::new);
      return super.modifyDeserializer(c, b, d);//@formatter:on
    }
  }

  static SystemDeserializer sys(JsonDeserializer<?> deserializer,
      FailableFunction<String, Entity, IOException> system,
      FailableFunction<String, Entity, IOException> builder) {
    return new SystemDeserializer(deserializer, system, builder);
  }

  static class SystemDeserializer extends JsonDeserializer<Entity> implements ResolvableDeserializer {

    final JsonDeserializer<?> deserializer;
    final FailableFunction<String, Entity, IOException> entity;
    final FailableFunction<String, Entity, IOException> system;

    private SystemDeserializer(JsonDeserializer<?> deserializer,
        FailableFunction<String, Entity, IOException> system,
        FailableFunction<String, Entity, IOException> entity) {
      this.deserializer = deserializer;
      this.system = system;
      this.entity = entity;
    }

    @Override
    public Entity deserialize(JsonParser jsonParser, DeserializationContext context) throws IOException {
      if (jsonParser.currentToken() == JsonToken.VALUE_NULL) return null;
      Entity entity;
      if (jsonParser.currentToken() == JsonToken.VALUE_STRING) {
        String text = jsonParser.getText();
        if (isEmpty(text)) return null;
        entity = this.entity.apply(text);
      } else {
        entity = (Entity) deserializer.deserialize(jsonParser, context);
      }
      String id = Optional.ofNullable(entity).map(Entity::getId).map(Id::getId).orElse(null);
      Entity systemEntity = isEmpty(id) ? null : system.apply(id);
      return systemEntity != null ? systemEntity : entity;
    }

    // for some reason you have to override this when modifying BeanDeserializer otherwise deserializing throws ex
    @Override
    public void resolve(DeserializationContext ctxt) throws JsonMappingException {
      ((ResolvableDeserializer) deserializer).resolve(ctxt);
    }
  }
}
