/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.serialization;

import static com.geotab.util.Util.listOf;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.geotab.model.entity.dvirlog.DVIRDefect;
import com.geotab.model.entity.dvirlog.DefectRemark;
import com.geotab.model.entity.dvirlog.RepairStatusType;
import com.geotab.model.entity.group.Defect;
import com.geotab.model.entity.user.User;
import com.geotab.util.DateTimeUtil;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.List;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@SuppressWarnings("AbbreviationAsWordInName")
public class DVIRDefectDeserializer extends JsonDeserializer<DVIRDefect> {

  @Override
  public DVIRDefect deserialize(JsonParser jp, DeserializationContext dc) throws IOException {
    ObjectCodec parserCodec = jp.getCodec();
    JsonNode node = parserCodec.readTree(jp);

    if (node.isObject()) {
      String id = node.get("id") == null ? null : node.get("id").textValue();
      RepairStatusType repairStatus = node.get("repairStatus") == null ? null
          : parserCodec.treeToValue(node.get("repairStatus"), RepairStatusType.class);
      User repairUser = node.get("repairUser") == null ? null
          : parserCodec.treeToValue(node.get("repairUser"), User.class);
      List<DefectRemark> defectRemarks = node.get("defectRemarks") == null ? null
          : listOf(parserCodec.treeToValue(node.get("defectRemarks"), DefectRemark[].class));
      LocalDateTime repairDateTime = node.get("repairDateTime") == null ? null
          : DateTimeUtil.localDateTimeFromString(node.get("repairDateTime").textValue());

      Defect defect = DefectDeserializer.deserializeWithChildren(node.get("defect"));

      return DVIRDefect.builder()
          .id(id)
          .repairStatus(repairStatus)
          .repairUser(repairUser)
          .repairDateTime(repairDateTime)
          .defect(defect)
          .defectRemarks(defectRemarks)
          .build();
    }

    return null;
  }
}
