/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.serialization;

import static com.geotab.model.entity.device.Device.HISTORIC_SERIAL_NUMBER;
import static com.geotab.model.entity.device.Device.UNTRACKED_ASSET_PRODUCT_ID;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.A1;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.CUSTOM_DEVICE;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.CUSTOM_VEHICLE_DEVICE;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO10;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO2;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO3;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO4;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO4V3;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO5;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO6;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO7;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO8;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO9;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO9B;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.GO_DRIVE_DEVICE;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.NONE;
import static com.geotab.model.entity.device.DeviceTypeNameConstants.OLD_GEOTAB;
import static com.geotab.model.serialization.ApiCustomDeserializerModifier.defaultDeserialize;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.geotab.model.entity.device.CustomDevice;
import com.geotab.model.entity.device.CustomVehicleDevice;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.device.DeviceType;
import com.geotab.model.entity.device.Go10;
import com.geotab.model.entity.device.Go4v3;
import com.geotab.model.entity.device.Go5;
import com.geotab.model.entity.device.Go6;
import com.geotab.model.entity.device.Go7;
import com.geotab.model.entity.device.Go8;
import com.geotab.model.entity.device.Go9;
import com.geotab.model.entity.device.Go9b;
import com.geotab.model.entity.device.GoDriveDevice;
import com.geotab.model.entity.device.GoLegacy;
import com.geotab.model.entity.device.NoDevice;
import com.geotab.model.entity.device.UntrackedAsset;
import com.geotab.util.Util;
import java.io.IOException;

public class DeviceDeserializer extends JsonDeserializer<Device> {

  private static final String DEVICE_TYPE_ATTRIBUTE_NAME = "deviceType";
  private static final String SERIAL_NUMBER_ATTRIBUTE_NAME = "serialNumber";
  private static final String PRODUCT_ID_ATTRIBUTE_NAME = "productId";

  @Override
  public Device deserialize(JsonParser jp, DeserializationContext c) throws IOException {
    ObjectCodec parserCodec = jp.getCodec();
    JsonNode node = parserCodec.readTree(jp);

    if (node.isTextual()) return NoDevice.getInstance();

    return hasDeviceType(node)
        ? byDeviceType(parserCodec, c, node)
        : bySerialNumberAndProductId(parserCodec, c, node);
  }

  private boolean hasDeviceType(JsonNode node) {
    return node.get(DEVICE_TYPE_ATTRIBUTE_NAME) != null
        && !node.get(DEVICE_TYPE_ATTRIBUTE_NAME).isNull()
        && DeviceType.findByName(node.get(DEVICE_TYPE_ATTRIBUTE_NAME).textValue()) != null;
  }

  private Device byDeviceType(ObjectCodec pc, DeserializationContext c, JsonNode n) throws IOException {
    switch (n.get(DEVICE_TYPE_ATTRIBUTE_NAME).asText()) {
      case OLD_GEOTAB:
      case GO2:
      case GO3:
      case GO4:
        return defaultDeserialize(pc, c, n, GoLegacy.class);
      case GO4V3:
        return defaultDeserialize(pc, c, n, Go4v3.class);
      case GO5:
        return defaultDeserialize(pc, c, n, Go5.class);
      case GO6:
        return defaultDeserialize(pc, c, n, Go6.class);
      case GO7:
        return defaultDeserialize(pc, c, n, Go7.class);
      case GO8:
        return defaultDeserialize(pc, c, n, Go8.class);
      case GO9:
        return defaultDeserialize(pc, c, n, Go9.class);
      case GO9B:
        return defaultDeserialize(pc, c, n, Go9b.class);
      case GO10:
        return defaultDeserialize(pc, c, n, Go10.class);
      case GO_DRIVE_DEVICE:
        return defaultDeserialize(pc, c, n, GoDriveDevice.class);
      case CUSTOM_DEVICE:
        return defaultDeserialize(pc, c, n, CustomDevice.class);
      case CUSTOM_VEHICLE_DEVICE:
        return defaultDeserialize(pc, c, n, CustomVehicleDevice.class);
      case A1:
        return defaultDeserialize(pc, c, n, com.geotab.model.entity.device.A1.class);
      case NONE:
        return defaultDeserialize(pc, c, n, UntrackedAsset.class);
      default:
        return defaultDeserialize(pc, c, n, Device.class);
    }
  }

  private Device bySerialNumberAndProductId(ObjectCodec oc, DeserializationContext c, JsonNode n) throws IOException {

    if (n.get(SERIAL_NUMBER_ATTRIBUTE_NAME) == null
        || n.get(SERIAL_NUMBER_ATTRIBUTE_NAME).isNull()
        || !n.get(SERIAL_NUMBER_ATTRIBUTE_NAME).isTextual()) {
      return defaultDeserialize(oc, c, n, Device.class);
    }

    String serialNumber = n.get(SERIAL_NUMBER_ATTRIBUTE_NAME).asText();

    final int productId;
    if (Util.isEmpty(serialNumber)) {
      productId = UNTRACKED_ASSET_PRODUCT_ID;
    } else if (HISTORIC_SERIAL_NUMBER.equals(serialNumber)
        && n.get(PRODUCT_ID_ATTRIBUTE_NAME) != null
        && !n.get(PRODUCT_ID_ATTRIBUTE_NAME).isNull()) {
      productId = n.get(PRODUCT_ID_ATTRIBUTE_NAME).asInt();
    } else {
      productId = Device.productIdFromSerialNumber(serialNumber);
    }

    DeviceType deviceType = Device.deviceTypeFromProductId(productId);
    Device device = defaultDeserialize(oc, c, n, deviceType.getType());
    if (device != null && device.getProductId() == null) {
      device.setProductId(productId);
    }

    return device;
  }
}
