/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.serialization;

import static java.lang.Integer.parseInt;
import static java.lang.Long.parseLong;

import com.geotab.util.Util;
import java.time.Duration;

/**
 * Serialization helper used to convert {@link Duration} to and from {@link String}.
 *
 * </p>Expected {@link Duration} pattern is [-][d.]hh:mm:ss[.fffffff] , as explained in
 * https://docs.microsoft.com/en-us/dotnet/standard/base-types/standard-timespan-format-strings .
 */
public interface DurationSerializationUtil {

  static Duration durationFromString(String str) {
    if (Util.isEmpty(str)) return null;
    if (!str.matches("^[-]?(\\d+[.])?\\d{1,2}:\\d{1,2}:\\d{1,2}([.]\\d{1,7})?$")) throw new IllegalArgumentException(
        "Unexpected duration pattern for value " + str + ", expected [-][d.]hh:mm:ss[.fffffff]");
    String[] parts = str.split(":");
    String[] secondsParts = parts[2].split("\\.");

    Duration out = Duration.ZERO;
    // negation
    boolean isNegated = false;
    if (parts[0].startsWith("-")) {
      parts[0] = parts[0].substring(1);
      isNegated = true;
    }
    // days
    int dotIndex = parts[0].indexOf('.');
    if (dotIndex > -1) {
      String[] dayParts = parts[0].split("\\.");
      out = out.plusDays(parseInt(dayParts[0]));
      parts[0] = parts[0].substring(dotIndex + 1);
    }
    // time
    out = out.plusHours(parseInt(parts[0])).plusMinutes(parseInt(parts[1])).plusSeconds(parseInt(secondsParts[0]));
    // nanos
    if (secondsParts.length > 1) { // in case there are less the 7 digits for ticks, then append zeros up to 7 digits
      out = out.plusNanos(parseLong(String.format("%1$-7s", secondsParts[1]).replaceAll(" ", "0")) * 100);
    }
    return isNegated ? out.negated() : out;
  }

  static String durationToString(Duration duration) {
    if (duration == null) return null;
    final long seconds = Math.abs(duration.getSeconds());
    final long nano = Math.abs(duration.getNano());
    final long days = seconds / 86400;

    StringBuilder out = new StringBuilder();
    if (duration.isNegative()) out.append("-");
    if (days > 0) out.append(days).append('.');
    out.append(String.format("%02d:%02d:%02d", seconds % 86400 / 3600, seconds % 3600 / 60, seconds % 60));
    if (nano > 0) out.append('.').append(String.format("%07d", nano / 100));
    return out.toString();
  }
}
