/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.api;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.geotab.http.request.AuthenticatedRequest;
import com.geotab.http.request.BaseRequest;
import com.geotab.http.request.MultiCallRequest;
import com.geotab.http.request.param.AuthenticatedParameters;
import com.geotab.http.request.param.EntityParameters;
import com.geotab.http.request.param.GetFeedParameters;
import com.geotab.http.request.param.SearchParameters;
import com.geotab.http.response.BaseResponse;
import com.geotab.model.FeedResult;
import com.geotab.model.Id;
import com.geotab.model.entity.Entity;
import com.geotab.model.login.Credentials;
import com.geotab.model.login.LoginResult;
import com.geotab.model.search.IdSearch;
import com.geotab.model.search.Search;
import java.io.Closeable;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import org.jetbrains.annotations.Nullable;

/**
 * Used to make API calls against a MyGeotab web server. It makes it easy to invoke the various methods and receive the
 * results. It also automates some tasks such as handling a database that was moved from one server to another or
 * credentials expiring.
 *
 * <p>All {@code callXxx} methods, if the API is not yet authenticated, will issue an authenticate() call automatically
 * before making the actual call. If the user session expires, it will try to re-authenticate using the credentials
 * provided initially.
 */
public interface Api extends Closeable {

  /**
   * <i>Add</i> method name.
   *
   * @deprecated use {@link #callAdd} instead.
   */
  String Add = "Add";

  /**
   * <i>Get</i> method name.
   *
   * @deprecated use {@link #callGet} instead.
   */
  String Get = "Get";

  /**
   * <i>GetCountOf</i> method name.
   *
   * @deprecated use {@link #callGetCountOf} instead.
   */
  String GetCountOf = "GetCountOf";

  /**
   * <i>GetFeed</i> method name.
   *
   * @deprecated use {@link #callGetFeed} instead.
   */
  String GetFeed = "GetFeed";

  /**
   * <i>Remove</i> method name.
   *
   * @deprecated use {@link #callRemove} instead.
   */
  String Remove = "Remove";

  /**
   * <i>Set</i> method name.
   *
   * @deprecated use {@link #callSet} instead.
   */
  String Set = "Set";

  /**
   * Authenticates a user and provides a {@link LoginResult} if successful. It contains the {@link Credentials} property
   * that can be used for further API calls. A result of LoginResult.Path = "ThisServer" occurs when the user is found
   * on the current server. Otherwise, a server name is returned and the client must redirect to this new server name.
   *
   * <p>Maximum 10 Authentication requests per minute, per user.
   *
   * @throws com.geotab.http.exception.InvalidUserException Invalid user or password.
   * @throws com.geotab.http.exception.DbUnavailableException Database unavailable.
   * @throws com.geotab.http.exception.OverLimitException Limit overflow.
   */
  LoginResult authenticate();

  /**
   * Check if the API is authenticated.
   *
   * <p>Should be authenticated only after explicit call to {@link Api#authenticate()} or any of the {@code callXxx}
   * methods.
   */
  boolean isAuthenticated();

  /**
   * Makes a custom method API call specifying the response type. Always try to use one of the specific method variants
   * * ({@link #callGet}, {@link #callGetFeed}, {@link #callAdd}, {@code call{Method}}, etc.) instead.
   *
   * @deprecated use {@link #callResult} instead and specify the result type instead of the response type.
   */
  <O extends BaseResponse<T>, T> Optional<T> call(AuthenticatedRequest<?> in, Class<O> outT);

  /**
   * Makes a custom method API call specifying the result type. Always try to use one of the specific method variants
   * ({@link #callGet}, {@link #callGetFeed}, {@link #callAdd}, {@code call{Method}}, etc.) instead.
   */
  <T> Optional<T> callResult(AuthenticatedRequest<?> in, Class<T> outT);

  /** Makes a custom method API call specifying the element type of list result type. */
  <T> Optional<List<T>> callResultList(AuthenticatedRequest<?> in, Class<T> outT);

  /**
   * Makes a web method call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  <P extends AuthenticatedParameters, R> Optional<R> callMethod(MethodDescriptor<P, R> method, P in);

  /** Makes an <i>Add</i> call. */
  Optional<Id> callAdd(AuthenticatedRequest<?> in);

  /** Makes an <i>Add</i> call. */
  default Optional<Id> callAdd(AuthenticatedParameters in) {
    return callAdd(AuthenticatedRequest.authRequestBuilder().method(Add).params(in).build());
  }

  /**
   * Makes a type-safe <i>Add</i> call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity> Optional<Id> callAdd(EntityDescriptor<E, ?> type, E entity) {
    return callAdd(EntityParameters.entityParamsBuilder().typeName(type.name()).entity(entity).build());
  }

  /** Makes a <i>Get</i> call. */
  <T extends Entity> Optional<List<T>> callGet(AuthenticatedRequest<?> in, Class<T> outT);

  /** Makes a <i>Get</i> call. */
  default <T extends Entity> Optional<List<T>> callGet(SearchParameters<?> in, Class<T> outT) {
    return callGet(AuthenticatedRequest.authRequestBuilder().method(Get).params(in).build(), outT);
  }

  /**
   * Makes a type-safe <i>Get</i> call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity, S extends Search> Optional<List<E>> callGet(EntityDescriptor<E, S> type, @Nullable S search) {
    return callGet(type, search, null);
  }

  /**
   * Makes a type-safe <i>Get</i> call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity, S extends Search> Optional<List<E>> callGet(EntityDescriptor<E, S> type, @Nullable S search,
      @Nullable Integer resultLimit) {
    return callGet(SearchParameters.searchParamsBuilder().typeName(type.name()).search(search).resultsLimit(resultLimit).build(), type.type);
  }

  /**
   * Makes a type-safe <i>Get</i> call to query all entities.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity> Optional<List<E>> callGetAll(EntityDescriptor<E, ?> type) {
    return callGet(type, null);
  }

  /**
   * Makes a type-safe <i>Get</i> call to query by ID.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity> Optional<E> callGetById(EntityDescriptor<E, ?> type, String id) {
    return callGet(SearchParameters.searchParamsBuilder().typeName(type.name()).search(new IdSearch(id)).build(), type.type)
        .map(o -> {
          if (o.isEmpty()) return null;
          if (o.size() > 1) throw new IllegalStateException("Query by ID returned more than one result");
          return o.get(0);
        });
  }

  /** Makes a <i>GetCountOf</i> call. */
  Optional<Integer> callGetCountOf(AuthenticatedRequest<?> in);

  /** Makes a <i>GetCountOf</i> call. */
  default Optional<Integer> callGetCountOf(SearchParameters<?> in) {
    return callGetCountOf(AuthenticatedRequest.authRequestBuilder().method(GetCountOf).params(in).build());
  }

  /**
   * Makes a type-safe <i>GetCountOf</i> call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity, S extends Search> Optional<Integer> callGetCountOf(EntityDescriptor<E, S> type, S search) {
    return callGetCountOf(SearchParameters.searchParamsBuilder().typeName(type.name()).search(search).build());
  }

  /** Makes a <i>GetFeed</i> call. */
  <T extends Entity> Optional<FeedResult<T>> callGetFeed(AuthenticatedRequest<?> in, Class<T> outT);

  /** Makes a <i>GetFeed</i> call. */
  default <T extends Entity> Optional<FeedResult<T>> callGetFeed(GetFeedParameters<?> in, Class<T> outT) {
    return callGetFeed(AuthenticatedRequest.authRequestBuilder().method(GetFeed).params(in).build(), outT);
  }

  /**
   * Makes a type-safe <i>GetFeed</i> call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity, S extends Search> Optional<List<E>> callGetFeed(EntityDescriptor<E, S> type, S search,
      @Nullable String fromVersion, @Nullable Integer resultLimit) {
    return callGet(GetFeedParameters.getFeedParamsBuilder().typeName(type.name()).search(search)
        .fromVersion(fromVersion).resultsLimit(resultLimit).build(), type.type);
  }

  /** Makes a <i>Set</i> call. */
  void callSet(AuthenticatedRequest<?> in);

  /** Makes a <i>Set</i> call. */
  default void callSet(AuthenticatedParameters in) {
    callSet(AuthenticatedRequest.authRequestBuilder().method(Set).params(in).build());
  }

  /**
   * Makes a type-safe <i>Set</i> call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity> void callSet(EntityDescriptor<E, ?> type, E entity) {
    callSet(EntityParameters.entityParamsBuilder().typeName(type.name()).entity(entity).build());
  }

  /** Makes a <i>Remove</i> call. */
  void callRemove(AuthenticatedRequest<?> in);

  /** Makes a <i>Remove</i> call. */
  default void callRemove(AuthenticatedParameters in) {
    callRemove(AuthenticatedRequest.authRequestBuilder().method(Remove).params(in).build());
  }

  /**
   * Makes a type-safe <i>Remove</i> call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  default <E extends Entity, S extends Search> void callRemove(EntityDescriptor<E, S> type, E entity) {
    callRemove(EntityParameters.entityParamsBuilder().typeName(type.name()).entity(entity).build());
  }

  /**
   * Makes a type-safe API multi-call.
   *
   * <p>Response type needs to be constructed based on the multi-call response types expected. The
   * API "result" is going to be an array, where each item is the result of the corresponding call.
   *
   * @param in MultiCallRequest request.
   * @param outT Response type class, used to deserialize the response.
   * @return The results for the multi-calls.
   * @deprecated use {{@link #buildMultiCall()}} instead.
   */
  <O extends BaseResponse<T>, T> Optional<T> multiCall(MultiCallRequest in, Class<O> outT);

  /**
   * Makes a uniform type-safe API multi-call.
   *
   * <p>All requests must return the same type. This method combine all responses sequentially into a single list. E.g.
   * if 3 device search requests are made and each request returns 2 devices a single list with 6 devices is returned.
   *
   * <p><b>WARNING:</b> Experimental API.
   *
   * @param calls List of requests. All request must have the same result type.
   * @param outT The common type of all requests.
   * @return A flat map of all the request.
   * @deprecated use {{@link #buildMultiCall()}} instead.
   */
  <T> Optional<List<T>> uniformMultiCall(List<? extends BaseRequest<?>> calls, Class<T> outT);

  /**
   * Builds a type-safe API multi-call. Accumulate request using {@link MultiCallBuilder#callResult} and finalize it
   * calling {@link MultiCallBuilder#execute}. The builder can not be reused and {@link MultiCallBuilder#execute} is
   * always the last call.
   *
   * <p><b>WARNING:</b> Experimental API.
   */
  MultiCallBuilder buildMultiCall();

  interface MultiCallBuilder {

    /** Custom Http Headers to be set on the multi-call request. */
    MultiCallBuilder httpHeaders(Map<String, Object> httpHeaders);

    /**
     * Makes a custom method API call specifying the response type. Always try to use one of the specific method
     * variants ({@link #callGet}, {@link #callGetFeed}, {@link #callAdd}, {@code call{Method}}, etc.) instead. The
     * supplier cannot be called before the builder is executed. Once the execution finalizes successfully, the supplier
     * will return the response result.
     *
     * @deprecated use {@link #callResult} instead and specify the result type instead of the response type.
     */
    <O extends BaseResponse<T>, T> Supplier<T> call(AuthenticatedRequest<?> in, Class<O> outT);

    /**
     * Makes a custom method API call specifying the result type. Always try to use one of the specific method variants
     * ({@link #callGet}, {@link #callGetFeed}, {@link #callAdd}, {@code call{Method}}, etc.) instead. The supplier
     * cannot be called before the builder is executed. Once the execution finalizes successfully, the supplier will
     * return the response result.
     */
    <T> Supplier<T> callResult(AuthenticatedRequest<?> in, Class<T> outT);

    /**
     * Makes a custom method API call specifying the element type of list result type. The supplier cannot be called
     * before the builder is executed. Once the execution finalizes successfully, the supplier will return the response
     * result.
     */
    <T> Supplier<List<T>> callResultList(AuthenticatedRequest<?> in, Class<T> outT);

    /**
     * Makes a web method call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    <P extends AuthenticatedParameters, R> Supplier<R> callMethod(MethodDescriptor<P, R> method, P in);

    /**
     * Makes an <i>Add</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    Supplier<Id> callAdd(AuthenticatedRequest<?> in);

    /**
     * Makes an <i>Add</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default Supplier<Id> callAdd(AuthenticatedParameters in) {
      return callAdd(AuthenticatedRequest.authRequestBuilder().method(Add).params(in).build());
    }

    /**
     * Makes a type-safe <i>Add</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <E extends Entity> Supplier<Id> callAdd(EntityDescriptor<E, ?> type, E entity) {
      return callAdd(EntityParameters.entityParamsBuilder().typeName(type.name()).entity(entity).build());
    }

    /**
     * Makes a <i>Get</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    <T extends Entity> Supplier<List<T>> callGet(AuthenticatedRequest<?> in, Class<T> outT);

    /**
     * Makes a <i>Get</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <T extends Entity> Supplier<List<T>> callGet(SearchParameters<?> in, Class<T> outT) {
      return callGet(AuthenticatedRequest.authRequestBuilder().method(Get).params(in).build(), outT);
    }

    /**
     * Makes a type-safe <i>Get</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <E extends Entity, S extends Search> Supplier<List<E>> callGet(EntityDescriptor<E, S> type, @Nullable S search) {
      return callGet(type, search, null);
    }

    /**
     * Makes a type-safe <i>Get</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <E extends Entity, S extends Search> Supplier<List<E>> callGet(EntityDescriptor<E, S> type, @Nullable S search,
        @Nullable Integer resultLimit) {
      return callGet(SearchParameters.searchParamsBuilder().typeName(type.name()).search(search).resultsLimit(resultLimit).build(), type.type);
    }

    /**
     * Makes a type-safe <i>Get</i> call to query all entities. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <E extends Entity> Supplier<List<E>> callGetAll(EntityDescriptor<E, ?> type) {
      return callGet(type, null);
    }

    /**
     * Makes a type-safe <i>Get</i> call to query by ID. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <E extends Entity> Supplier<E> callGetById(EntityDescriptor<E, ?> type, String id) {
      return () -> {
        List<E> o = callGet(SearchParameters.searchParamsBuilder().typeName(type.name()).search(new IdSearch(id)).build(), type.type).get();
        if (o.isEmpty()) return null;
        if (o.size() > 1) throw new IllegalStateException("Query by ID returned more than one result");
        return o.get(0);
      };
    }

    /**
     * Makes a <i>GetCountOf</i> call. The supplier cannot be called before the builder is executed. Once the
     * execution finalizes successfully, the supplier will return the response result.
     */
    Supplier<Integer> callGetCountOf(AuthenticatedRequest<?> in);

    /**
     * Makes a <i>GetCountOf</i> call. The supplier cannot be called before the builder is executed. Once the
     * execution finalizes successfully, the supplier will return the response result.
     */
    default Supplier<Integer> callGetCountOf(SearchParameters<?> in) {
      return callGetCountOf(AuthenticatedRequest.authRequestBuilder().method(GetCountOf).params(in).build());
    }

    /**
     * Makes a type-safe <i>GetCountOf</i> API call. The supplier cannot be called before the builder is executed. Once the
     * execution finalizes successfully, the supplier will return the response result.
     */
    default <E extends Entity, S extends Search> Supplier<Integer> callGetCountOf(EntityDescriptor<E, S> type, S search) {
      return callGetCountOf(SearchParameters.searchParamsBuilder().typeName(type.name()).search(search).build());
    }

    /**
     * Makes a <i>GetFeed</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    <T extends Entity> Supplier<FeedResult<T>> callGetFeed(AuthenticatedRequest<?> in, Class<T> outT);

    /**
     * Makes a <i>GetFeed</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <T extends Entity> Supplier<FeedResult<T>> callGetFeed(GetFeedParameters<?> in, Class<T> outT) {
      return callGetFeed(AuthenticatedRequest.authRequestBuilder().method(GetFeed).params(in).build(), outT);
    }

    /**
     * Makes a <i>GetFeed</i> call. The supplier cannot be called before the builder is executed. Once the execution
     * finalizes successfully, the supplier will return the response result.
     */
    default <E extends Entity, S extends Search> Supplier<List<E>> callGetFeed(EntityDescriptor<E, S> type, S search,
        @Nullable String fromVersion, @Nullable Integer resultLimit) {
      return callGet(GetFeedParameters.getFeedParamsBuilder().typeName(type.name()).search(search)
          .fromVersion(fromVersion).resultsLimit(resultLimit).build(), type.type);
    }

    /** Makes a <i>Set</i> call. */
    void callSet(AuthenticatedRequest<?> in);

    /** Makes a <i>Set</i> call. */
    default void callSet(AuthenticatedParameters in) {
      callSet(AuthenticatedRequest.authRequestBuilder().method(Set).params(in).build());
    }

    /** Makes a type-safe <i>Set</i> call. */
    default <E extends Entity> void callSet(EntityDescriptor<E, ?> type, E entity) {
      callSet(EntityParameters.entityParamsBuilder().typeName(type.name()).entity(entity).build());
    }

    /** Makes a <i>Remove</i> call. */
    void callRemove(AuthenticatedRequest<?> in);

    /** Makes a <i>Remove</i> call. */
    default void callRemove(AuthenticatedParameters in) {
      callRemove(AuthenticatedRequest.authRequestBuilder().method(Remove).params(in).build());
    }

    /** Makes a type-safe <i>Remove</i> call. */
    default <E extends Entity, S extends Search> void callRemove(EntityDescriptor<E, S> type, E entity) {
      callRemove(EntityParameters.entityParamsBuilder().typeName(type.name()).entity(entity).build());
    }

    /**
     * Execute the multi-call request. Once called, suppliers will be ready and the builder won't be usable anymore.
     */
    void execute();

  }

  static <P extends AuthenticatedParameters, R> MethodDescriptor<P, R> method(String name, Class<? super R> result, Class<?>... parameterClasses) {
    return new MethodDescriptor<>(name, result, parameterClasses);
  }

  class MethodDescriptor<P extends AuthenticatedParameters, R> {
    public final String name;
    public final Class<? super R> result;
    public final Class<?>[] parameterClasses;

    public MethodDescriptor(String name, Class<? super R> result, Class<?>... parameterClasses) {
      this.name = name;
      this.result = result;
      this.parameterClasses = parameterClasses;
    }

    /** Internal use only. */
    public JavaType resultType() {
      if (parameterClasses.length == 0) return TypeFactory.defaultInstance().constructType(result);
      return TypeFactory.defaultInstance().constructParametricType(result, parameterClasses);
    }

    @Override
    public String toString() {
      return name;
    }
  }

  static <E extends Entity, S extends Search> EntityDescriptor<E, S> entity(Class<E> type) {
    return new EntityDescriptor<>(type);
  }

  class EntityDescriptor<E extends Entity, S extends Search> {
    public final Class<E> type;

    public EntityDescriptor(Class<E> type) {
      this.type = type;
    }

    public String name() {
      return type.getSimpleName();
    }

    @Override
    public String toString() {
      return name();
    }
  }
}
