/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.http.exception;

import static com.geotab.util.Util.isNotEmpty;

import com.fasterxml.jackson.core.type.TypeReference;
import com.geotab.http.response.BaseResponse;
import com.geotab.model.error.JsonRpcError;
import com.geotab.model.error.JsonRpcErrorCode;
import com.geotab.model.error.JsonRpcErrorData;
import com.geotab.model.serialization.ApiJsonSerializer;
import java.io.IOException;
import java.util.Map;
import org.jetbrains.annotations.Nullable;

/**
 * Utility class which checks JSON response for error and transforms it to the corresponding exception.
 */
public final class ErrorHandler {

  /**
   * Checks if an error returned from MyGeotab.
   *
   * @param method   The method in MyGeotab API.
   * @param response The actual response from MyGeotab.
   */
  public static <R extends BaseResponse<?>> void checkForError(String method, R response) {
    checkForError(method, response != null ? response.getError() : null);
  }

  public static void checkForError(String method, @Nullable JsonRpcError error) {
    if (error == null || error.getData() == null) return;

    String formatted = String.format("Error response in call to '%s' [type='%s, message=%s]",
        method, error.getData().getType(), error.getMessage());

    RuntimeException exception = checkForError(formatted,
        new ServerInvokerJsonException(formatted, null, null, error.getData()),
        JsonRpcErrorCode.getByCode(error.getCode()), error.getData());

    if (exception instanceof JsonRpcErrorDataException) {
      ((JsonRpcErrorDataException) exception).setErrorData(error.getData());
    }

    throw exception;
  }

  static RuntimeException checkForError(String message, ServerInvokerJsonException ex, JsonRpcErrorCode code,
      JsonRpcErrorData data) {
    switch (code) {
      case INVALID_REQUEST:
        return new InvalidRequestException(message, ex);
      case DB_UNAVAILABLE_GENERAL_ERROR:
      case DB_UNAVAILABLE_CONNECTION_FAILURE:
      case DB_UNAVAILABLE_INITIALIZING:
      case DB_UNAVAILABLE_UNKNOWN_DATABASE:
      case DB_UNAVAILABLE_OPERATION_ABORTED:
        DbUnavailableState state = DbUnavailableState.NONE;
        if (isNotEmpty(data.getInfo())) {
          try {
            Map<String, DbUnavailableState> errorDataInfo = ApiJsonSerializer.getInstance().getObjectMapper()
                .readValue(data.getInfo(), new TypeReference<Map<String, DbUnavailableState>>() {});
            if (errorDataInfo.containsKey("state")) {
              state = errorDataInfo.get("state");
            }
          } catch (IOException e) {
            throw new IllegalStateException("Can not deserialize " + data.getInfo() + " to Map");
          }
        }
        return new DbUnavailableException(message, ex, state);
      case METHOD_NOT_FOUND:
        return new MissingMethodException(message, ex);
      case PARSE_ERROR:
        return new JsonRpcErrorDataException(message, ex);
      default:
        return checkForErrorByErrorDataType(message, ex, data);
    }
  }

  static RuntimeException checkForErrorByErrorDataType(String message, ServerInvokerJsonException ex,
      JsonRpcErrorData data) {
    switch (data.getType()) {
      case "InvalidMyAdminUserException":
        if (ex != null && ex.getInnerError() != null) {
          Map<String, String> innerError = ex.getInnerError();
          if (innerError.containsKey("message")) {
            String databaseName = innerError.get("message");
            if (isNotEmpty(databaseName)) {
              return new InvalidMyAdminUserException(databaseName);
            }
          }
        }
        return new InvalidMyAdminUserException();
      case "InvalidUserException":
        return new InvalidUserException();
      case "InactiveUserException":
        return new InactiveUserException(null);
      case "DbUnavailableException":
        return new DbUnavailableException(message, ex, DbUnavailableState.NONE);
      case "RelationViolatedException":
        return new RelationViolatedException(ex);
      case "GroupRelationViolatedException":
        return new GroupRelationViolatedException(ex);
      case "DuplicateException":
        return new DuplicateException(ex);
      case "PasswordPolicyViolationException":
        return new PasswordPolicyViolationException(message);
      case "RegistrationException":
        return new RegistrationException(message);
      case "OverLimitException":
        return new OverLimitException(message);
      case "InvalidRequestException":
        return new IllegalStateException("Inconsistent JSON-RPC error code");
      case "MethodNotFoundException":
      case "MissingMethodException":
        return new MissingMethodException(message, ex);
      case "MissingMemberException":
        return new MissingMemberException(message, ex);
      case "NullReferenceException":
        return new NullPointerException(message);
      case "ArgumentException":
      case "ArgumentOutOfRangeException":
        return new IllegalArgumentException(message, ex);
      case "JsonSerializerException":
      case "JSONSerializerException":
        return new JsonRpcErrorDataException(message, ex);
      case "InvalidCastException":
        return new ClassCastException(message);
      case "NotSupportedException":
        return new JsonRpcErrorDataException(message, ex);
      case "InvalidCertificateException":
        return new InvalidCertificateException(message, ex);
      default:
        return ex;
    }
  }
}
