/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.drawing;

import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * A Rectangle with a float X, Y, Width and Height.
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
@Builder
public class RectangleF {

  /**
   * An empty rectangle.
   */
  public static final RectangleF EMPTY = new RectangleF(0F, 0F, 0F, 0F);

  /**
   * The X coordinate.
   */
  private Float x;

  /**
   * The Y coordinate.
   */
  private Float y;

  /**
   * The width of the rectangle.
   */
  private Float width;

  /**
   * The height of the rectangle.
   */
  private Float height;

  /**
   * The left (X) of the rectangle.
   *
   * @return The X coordinate.
   */
  @JsonIgnore
  public Float getLeft() {
    return this.x;
  }

  /**
   * The right (X) of the rectangle.
   *
   * @return The X + Width coordinate.
   */
  @JsonIgnore
  public Float getRight() {
    return this.x + this.width;
  }

  /**
   * The top (Y) of the rectangle.
   *
   * @return The Y coordinate.
   */
  @JsonIgnore
  public Float getTop() {
    return this.y;
  }

  /**
   * The bottom (Y) of the rectangle.
   *
   * @return The Y  + Height coordinate.
   */
  @JsonIgnore
  public Float getBottom() {
    return this.y + this.height;
  }

  /**
   * The location (top left) of the rectangle.
   *
   * @return Top-left of the rectangle.
   */
  @JsonIgnore
  public PointF getLocation() {
    return new PointF(this.x, this.y);
  }

  /**
   * Gets a new rectangle that is offset from the current by the point p.
   *
   * @param point The offset point.
   * @return The new {@link RectangleF}.
   */
  @JsonIgnore
  public RectangleF offset(PointF point) {
    return new RectangleF(this.x + point.getX(), this.y + point.getY(), this.width, this.height);
  }

  /**
   * Gets a rectangle from left top right bottom.
   *
   * @param left   The left.
   * @param top    The top.
   * @param right  The right.
   * @param bottom The bottom.
   * @return The new {@link RectangleF}.
   */
  public static RectangleF fromLtrb(float left, float top, float right, float bottom) {
    return new RectangleF(left, top, right - left, bottom - top);
  }


}
