/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.Id;
import com.geotab.model.serialization.IdAsStringSerializer;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;
import lombok.AccessLevel;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

/**
 * An {@link Entity} that has an id field.
 */
@Data
public abstract class Entity {

  @JsonSerialize(using = IdAsStringSerializer.class)
  private Id id;

  @Getter(AccessLevel.NONE)
  @Setter(AccessLevel.NONE)
  @EqualsAndHashCode.Exclude
  @ToString.Exclude
  private Map<String, Object> props;

  protected Entity() {
  }

  protected Entity(EntityBuilder<?, ?> b) {
    this.id = b.id;
  }

  /**
   * Returns true if the class is a system entity, false otherwise.
   */
  @JsonIgnore
  public boolean isSystemEntity() {
    return false;
  }

  /**
   * Experimental feature to set extra properties in entity models. Use only for exploratory or debugging purposes. All
   * official public data is exposed using strong typing.
   */
  @JsonAnySetter
  public void addExtraProperty(String k, Object v) {
    if (props == null) {
      props = new TreeMap<>();
    }
    props.put(k, v);
  }

  /**
   * Experimental feature to expose extra properties in entity models. Use only for exploratory or debugging purposes.
   * All official public data is exposed using strong typing.
   */
  @JsonAnyGetter
  public Map<String, Object> getExtraProperties() {
    return props == null ? Collections.emptyMap() : props;
  }

  public abstract static class EntityBuilder<C extends Entity, B extends EntityBuilder<C, B>> {

    private Id id;

    public B id(Id id) {
      this.id = id;
      return self();
    }

    public B id(String id) {
      this.id = new Id(id);
      return self();
    }

    protected abstract B self();

    public abstract C build();

    public String toString() {
      return "Entity.EntityBuilder(id=" + this.id + ")";
    }
  }
}
