/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.condition;

import com.geotab.model.entity.Entity;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.diagnostic.Diagnostic;
import com.geotab.model.entity.rule.Rule;
import com.geotab.model.entity.unitofmeasure.UnitOfMeasure;
import com.geotab.model.entity.user.Driver;
import com.geotab.model.entity.user.User;
import com.geotab.model.entity.worktime.WorkTime;
import com.geotab.model.entity.zone.Zone;
import com.geotab.model.entity.zone.type.ZoneType;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Conditions model the logic that govern a {@link Rule} and can apply to many different types of data and entities.
 * Conditions are structured in hierarchical tree. A condition's type (see {@link ConditionType}) defines the meaning of
 * each condition in the tree and can be an operator, special operator, data or an asset.
 *
 * <p>Depending on the type of condition, it can have a minimum of 0 and maximum of 1 entity
 * properties (Device, Driver, Diagnostic, WorkTime, Zone or ZoneType) defined per condition. Operator conditions (OR,
 * AND, >, <, ==, NOT) will not have any entity properties populated. Special Operator conditions evaluate against
 * special types of data such as Aux data, Zones, WorkHours, etc. and may have the entity property populated and/or a
 * child condition populated with a Data condition. Asset conditions will only have the asset entity property
 * populated.
 *
 * <p>The unit of measure for data is described either by the related {@link Diagnostic}'s
 * {@link UnitOfMeasure} or as follows:
 * <ul>
 *    <li>Distance: Meters (m)</li>
 *    <li>Speed: Kilometers Per Hour (km/h)</li>
 *    <li>Duration: Seconds</li>
 * </ul>
 *
 * <p>A tree of conditions can define simple or complex rules and can be very powerful.
 * Please take into consideration all possible consequences of a series of rules. Overly complex,
 * poorly written or an excessive number of rules can have undesirable performance effects.
 *
 * <p>This class is in development. Its public interface is a subject to change.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class Condition extends Entity {

  /**
   * Child condition(s) of this condition.
   */
  private List<Condition> children;

  /**
   * The {@link ConditionType} defines the meaning of this condition.
   */
  private ConditionType conditionType;

  /**
   * The sequence that defines the Condition's relative order in a list of Children Conditions of the parent Condition.
   */
  private Long sequence;

  /**
   * The specified value to evaluate against.
   */
  private Double value;

  /**
   * Specified {@link Device} associated with the condition.
   */
  private Device device;

  /**
   * The {@link Diagnostic} to compare the value of.
   */
  private Diagnostic diagnostic;

  /**
   * Specified {@link Driver} associated with the condition.
   */
  private User driver;

  /**
   * The {@link WorkTime} that the event must occur inside/outside of for the violation to occur.
   */
  private WorkTime workTime;

  /**
   * Specified {@link Zone} associated with the condition.
   */
  private Zone zone;

  /**
   * Specified {@link ZoneType}.
   */
  private ZoneType zoneType;

  /**
   * The name of the metric used.
   */
  private String unit;
}
