/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.condition;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.diagnostic.Diagnostic;
import com.geotab.model.entity.faultdata.FaultData;
import com.geotab.model.entity.group.Group;
import com.geotab.model.entity.logrecord.LogRecord;
import com.geotab.model.entity.rule.Rule;
import com.geotab.model.entity.statusdata.StatusData;
import com.geotab.model.entity.trip.Trip;
import com.geotab.model.entity.user.Driver;
import com.geotab.model.entity.worktime.WorkTime;
import com.geotab.model.entity.zone.Zone;
import com.geotab.model.entity.zone.type.ZoneType;
import com.geotab.model.serialization.HasName;

public enum ConditionType implements HasName {

  /**
   * Fallback when enumeration value is unknown.
   */
  @JsonEnumDefaultValue
  UNKNOWN("Unknown", -1),

  /**
   * None.
   */
  NONE("None", 0),

  /**
   * Operator: Condition for "And" operations. "And" conditions must have two or more children that will act as the
   * operands in the equation.
   */
  AND("And", 1),

  /**
   * Operator: Condition for "Or" operations. "Or" conditions must have two or more children that will act as the
   * operands in the equation.
   */
  OR("Or", 2),

  /**
   * Data: Evaluate the condition against active {@link FaultData}. Include a child diagnostic condition with related
   * fault {@link Diagnostic} or NoDiagnostic to detect any {@link FaultData}.
   */
  FAULT("Fault", 5),

  /**
   * Data: Evaluate the condition against active or inactive {@link FaultData}. Include a child diagnostic condition
   * with related fault {@link Diagnostic} or NoDiagnostic to detect any {@link FaultData}.
   */
  ACTIVE_OR_INACTIVE_FAULT("ActiveOrInactiveFault", 6),

  /**
   * Zone Operator: Evaluate if related the asset(s) are inside the {@link Zone} specified by this condition.
   */
  INSIDE_AREA("InsideArea", 10),

  /**
   * Zone Operator: Evaluate if related the asset(s) are inside the {@link Zone} specified by this condition.
   */
  OUTSIDE_AREA("OutsideArea", 11),

  /**
   * Data: Evaluate against the related asset(s) trip stop value.
   */
  STOP("Stop", 12),

  /**
   * Zone Operator: Evaluate if the related asset(s) are stopped inside the {@link Zone} specified in this condition.
   */
  ZONE_STOP("ZoneStop", 13),

  /**
   * Zone Operator: Evaluate if the related asset(s) are entering the bounds a {@link Zone} specified in this
   * condition.
   */
  ENTERING_AREA("EnteringArea", 14),

  /**
   * Zone Operator: Evaluate if the related asset(s) are exiting the bounds a {@link Zone} specified in this condition.
   */
  EXITING_AREA("ExitingArea", 15),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 1 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_1("Aux1", 16),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 2 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_2("Aux2", 17),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 3 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_3("Aux3", 18),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 4 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_4("Aux4", 19),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 5 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_5("Aux5", 20),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 6 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_6("Aux6", 21),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 7 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_7("Aux7", 22),

  /**
   * Auxiliary Operator: Evaluate against the asset(s) Auxiliary 8 value. Auxiliary conditions can have the value of 0
   * (False) or 1 (True).
   */
  AUX_8("Aux8", 23),

  /**
   * Ignition Operator: Evaluate against the asset(s) ignition value. Ignition conditions can have the value of 0 (Off)
   * or 1 (On).
   */
  IGNITION("Ignition", 24),

  /**
   * Value Operator: The duration of the child condition must continue to be true for longer than the value of this
   * condition in seconds.
   */
  DURATION_LONGER_THAN("DurationLongerThan", 25),

  /**
   * Value Operator: The duration of the child condition must continue to be true for no longer than the value of this
   * condition in seconds.
   */
  DURATION_SHORTER_THAN("DurationShorterThan", 26),

  /**
   * Value Operator: The duration of the child condition must continue to be true for a distance longer than the value
   * of this condition in km.
   */
  DISTANCE_LONGER_THAN("DistanceLongerThan", 27),

  /**
   * Value Operator: The duration of the child condition must continue to be true for no longer distance than the value
   * of this condition in km.
   */
  DISTANCE_SHORTER_THAN("DistanceShorterThan", 28),

  /**
   * Work Hours Operator: Occurs during the {@link Device}'s assigned {@link WorkTime}.
   */
  DEVICE_WORK_HOURS("DeviceWorkHours", 29),

  /**
   * Work Hours Operator: Occurs after the {@link Device}'s assigned {@link WorkTime}.
   */
  AFTER_DEVICE_WORK_HOURS("AfterDeviceWorkHours", 30),

  /**
   * Work Hours Operator: Occurs during the {@link WorkTime} assigned to this condition.
   */
  RULE_WORK_HOURS("RuleWorkHours", 31),

  /**
   * Work Hours Operator: Occurs after the {@link WorkTime} assigned to this condition.
   */
  AFTER_RULE_WORK_HOURS("AfterRuleWorkHours", 32),

  /**
   * Asset: Apply to the {@link Device} specified in this condition. This will take priority over the {@link Group}(s)
   * assigned to the {@link Rule}. When no asset condition is specified the rule will apply to all assets in the rule's
   * groups.
   */
  DEVICE("Device", 33),

  /**
   * Asset: Apply to the {@link Device} that the {@link Driver} specified in this condition is assigned to. This will
   * take priority over the {@link Group}(s) assigned to the {@link Rule}. When no asset condition is specified the rule
   * will apply to all assets in the rule's groups.
   */
  DRIVER("Driver", 34),

  /**
   * System: A method of reporting application exceptions.
   */
  APPLICATION_EXCEPTION("ApplicationException", 35),

  /**
   * Operator: Invert the results of the child condition(s).
   */
  INVERT_RESULT("InvertResult", 38),

  /**
   * Data: Evaluate the condition against {@link StatusData} related to a particular {@link Diagnostic}. This condition
   * will have the Diagnostic property populated and is used in conjunction with (as child of) an operator
   * (IsValueMoreThan, IsValueLessThan, IsValueEqualTo, AnyData).
   */
  FILTER_STATUS_DATA_BY_DIAGNOSTIC("FilterStatusDataByDiagnostic", 39),

  /**
   * Operator: The result of the child condition is greater than value of this condition.
   */
  IS_VALUE_MORE_THAN("IsValueMoreThan", 40),

  /**
   * Operator: The result of the child condition is less than value of this condition.
   */
  IS_VALUE_LESS_THAN("IsValueLessThan", 41),

  /**
   * Operator: The result of the child condition is equal to value of this condition.
   */
  IS_VALUE_EQUAL_TO("IsValueEqualTo", 42),

  /**
   * Operator: True when the expected duration of the child condition meets this conditions value in seconds.
   */
  EXPECTED_DURATION("ExpectedDuration", 44),

  /**
   * Operator: True when the expected distance of the child condition meets this conditions value in km.
   */
  EXPECTED_DISTANCE("ExpectedDistance", 45),

  /**
   * Data: The trip duration of the asset in seconds. Compare against this value using an operator. Example:
   * DurationLongerThan(value) - child of TripDuration condition.
   */
  TRIP_DURATION("TripDuration", 46),

  /**
   * Data: The speed of the asset in km/h. Compare against this value using an operator. Example: IsValueMoreThan(value)
   * - child of Speed condition.
   */
  SPEED("Speed", 47),

  /**
   * Data: The trip distance of the asset in km. Compare against this value using an operator. Example:
   * DistanceLongerThan(value) - child of TripDistance condition.
   */
  TRIP_DISTANCE("TripDistance", 48),

  /**
   * Data: The posted road speed of the road the asset is located on in km/h. Compare against this value using an
   * operator and comparing to speed.
   */
  SPEED_LIMIT("SpeedLimit", 50),

  /**
   * Data: Is the asset driving. Extract a sequence of values of +1 (at start of driving), -1 (at beginning of
   * stoppage), 0 (state unknown: usually occurs at start and the end of available span of the {@link LogRecord}(s)).
   */
  IS_DRIVING("IsDriving", 51),

  /**
   * Data: The posted road speed of the road the asset is located on in km/h. Compare against this value using an
   * operator and comparing to speed; excludes estimate speed values.
   */
  SPEED_LIMIT_EXCLUDING_ESTIMATES("SpeedLimitExcludingEstimates", 52),

  /**
   * Data: Time Between GPS points in seconds.
   */
  DURATION_BETWEEN_GPS("DurationBetweenGps", 53),

  /**
   * Data: Distance Between GPS points in meters.
   */
  DISTANCE_BETWEEN_GPS("DistanceBetweenGps", 54),

  /**
   * Data: The posted road speed of the road the asset is located on in km/h. Compare against this value using an
   * operator and comparing to speed, uses community speed data only.
   */
  SPEED_LIMIT_COMMUNITY("SpeedLimitCommunity", 55),

  /**
   * Data: The posted road speed of the road the asset is located on in km/h. Compare against this value using an
   * operator and comparing to speed, uses community speed data only; excludes estimate speed values.
   */
  SPEED_LIMIT_COMMUNITY_EXCLUDING_ESTIMATES("SpeedLimitCommunityExcludingEstimates", 56),

  /**
   * Data: The posted road speed of the road the asset is located on in km/h. Compare against this value using an
   * operator and comparing to speed, uses commercial speed data only.
   */
  SPEED_LIMIT_COMMERCIAL("SpeedLimitCommercial", 57),

  /**
   * Data: The posted road speed of the road the asset is located on in km/h. Compare against this value using an
   * operator and comparing to speed, uses commercial speed data only; excludes estimate speed values.
   */
  SPEED_LIMIT_COMMERCIAL_EXCLUDING_ESTIMATES("SpeedLimitCommercialExcludingEstimates", 58),

  /**
   * Data: Exception event for whenever DVIRDefect is detected.
   */
  DVIR_DEFECT("DVIRDefect", 60),

  /**
   * Data: No Pre or Post DVIR check is performed between working days.
   *
   * <p>NoDVIRCheck is obsolete as of v5.7.2002. Replaced by NoPreDVIRCheck and NoPostDVIRCheck.
   */
  NO_DVIR_CHECK("NoDVIRCheck", 61),

  /**
   * Data: Exception event for the entire duration of a {@link Trip}.
   */
  ENTIRE_TRIP("EntireTrip", 62),

  /**
   * Operator: Condition for "Inclusive" operations. "Inclusive" conditions must have two or more children that will act
   * as the operands in the equation.
   */
  INCLUSIVE("Inclusive", 63),

  /**
   * Data: The posted road speed of the road the asset is located on in km/h. Used as measurement and filtered by its
   * parent filters.
   */
  SPEED_LIMIT_AS_MEASUREMENT("SpeedLimitAsMeasurement", 64),

  /**
   * Data: No Pre or Post DVIR check is performed between working days.
   */
  NO_PRE_DVIR_CHECK("NoPreDVIRCheck", 65),

  /**
   * Data: No Pre or Post DVIR check is performed between working days.
   */
  NO_POST_DVIR_CHECK("NoPostDVIRCheck", 66),

  /**
   * Operator: The result of the child condition is greater than a percentage of the value of this condition.
   */
  IS_VALUE_MORE_THAN_PERCENT("IsValueMoreThanPercent", 67),

  /**
   * Operator: The result of the child condition is less than a percentage of the value of this condition.
   */
  IS_VALUE_LESS_THAN_PERCENT("IsValueLessThanPercent", 68),

  /**
   * Data: The average road speed of the road the asset is located on in km/h. Compare against this value using an
   * operator and comparing to speed.
   */
  AVERAGE_ROAD_SPEED("AverageRoadSpeed", 69),

  /**
   * Data: The posted road speed for trucks of the road the asset is located on in km/h. Compare against this value
   * using an operator and comparing to speed.
   */
  SPEED_LIMIT_TRUCK("SpeedLimitTruck", 70),

  /**
   * Data: The posted road speed for trucks of the road the asset is located on in km/h. Compare against this value
   * using an operator and comparing to speed; excludes estimate speed values.
   */
  SPEED_LIMIT_TRUCK_EXCLUDING_ESTIMATES("SpeedLimitTruckExcludingEstimates", 71),

  /**
   * Data: The posted road speed for trucks of the road the asset is located on in km/h. Compare against this value
   * using an operator and comparing to speed, uses community speed data only.
   */
  SPEED_LIMIT_TRUCK_COMMUNITY("SpeedLimitTruckCommunity", 72),

  /**
   * Data: The posted road speed for trucks of the road the asset is located on in km/h. Compare against this value
   * using an operator and comparing to speed, uses community speed data only; excludes estimate speed values.
   */
  SPEED_LIMIT_TRUCK_COMMUNITY_EXCLUDING_ESTIMATES("SpeedLimitTruckCommunityExcludingEstimates", 73),

  /**
   * Data: The posted road speed for trucks of the road the asset is located on in km/h. Compare against this value
   * using an operator and comparing to speed, uses commercial speed data only.
   */
  SPEED_LIMIT_TRUCK_COMMERCIAL("SpeedLimitTruckCommercial", 74),

  /**
   * Data: The posted road speed for trucks of the road the asset is located on in km/h. Compare against this value
   * using an operator and comparing to speed, uses commercial speed data only; excludes estimate speed values.
   */
  SPEED_LIMIT_TRUCK_COMMERCIAL_EXCLUDING_ESTIMATES("SpeedLimitTruckCommercialExcludingEstimates", 75),

  /**
   * Zone Operator: Evaluate if related the asset(s) are inside the {@link ZoneType} specified by this condition. It
   * must evaluate each {@link Zone} in the ZoneType independently.
   */
  INSIDE_EACH_AREA("InsideEachArea", 76);

  private final String name;
  private final int code;

  ConditionType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }
}
