package com.geotab.model.entity.device;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.geotab.util.DeviceDefaultsUtil;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * A custom third-party automotive vehicle device that is used in MyGeotab. This is used for extensibility and is based
 * on the {@link CustomDevice} object.
 */
@Data
@NoArgsConstructor
@SuperBuilder(builderMethodName = "customVehicleDeviceBuilder")
public class CustomVehicleDevice extends CustomDevice implements VehicleIdentificationNumberAware,
    OdometerAdjustmentProviderAware, EngineHoursAdjustmentProviderAware, LicensableAware {

  /**
   * Vehicle Identification Number (VIN) of the vehicle associated with the device. Maximum length [50] Default [""].
   */
  private String vehicleIdentificationNumber;

  /**
   * Engine Vehicle Identification Number from the engine of the vehicle associated with the device.
   */
  private String engineVehicleIdentificationNumber;

  /**
   * The Odometer for the device.
   */
  private Double odometer;

  /**
   * A value used to correct the odometer value received from the engine. Default [1].
   */
  private Float odometerFactor;

  /**
   * The offset to be applied odometer reported by the engine computer. Default [0].
   */
  private Double odometerOffset;

  /**
   * The Engine Hours for the device. Default [0].
   */
  private Double engineHours;

  /**
   * The offset to be applied engine hours data reported by the engine computer. Default [0].
   */
  private Integer engineHourOffset;

  /**
   * The vehicle license plate details of the vehicle associated with the device. Maximum length [50] Default [""].
   */
  private String licensePlate;

  /**
   * The state or province of the vehicle associated with the device. Maximum length [50] Default [""].
   */
  private String licenseState;

  /**
   * Toggle that determines whether or not a device is under automatic vehicle management. Default [false].
   */
  @JsonProperty("pinDevice")
  private Boolean pinDevice;

  /**
   * Disable or enable sleeper berth.
   */
  private Boolean disableSleeperBerth;

  /**
   * A toggle that represents automatic generation of DutyStatusLogs for a {@link Device}.
   */
  private HosOption autoHos;

  /**
   * The capacity of all usable fuel tanks in litres. Default [null].
   */
  private Double fuelTankCapacity;

  @Override
  public void populateDefaults() {
    super.populateDefaults();
    DeviceDefaultsUtil.addCustomVehicleDeviceDefaults(this);
  }

  @JsonIgnore
  public boolean isAutomaticVehicleManagementEnabled() {
    return pinDevice == null || !pinDevice;
  }

  {
    setDeviceType(DeviceType.CUSTOM_VEHICLE_DEVICE);
  }

  @Override
  public Boolean isPinDevice() {
    return getPinDevice();
  }
}
