package com.geotab.model.entity.device;

import static com.geotab.util.Util.listOf;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.geotab.util.Util;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The plan a device is on.
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
@Builder
public class DeviceFlags {

  /**
   * The {@link VehicleFeatureCategory}s that relate to Garmin.
   */
  public static final List<VehicleFeatureCategory> GARMIN_FEATURES = listOf(
      VehicleFeatureCategory.GARMIN, VehicleFeatureCategory.GARMIN_HOS);

  /**
   * The {@link VehicleFeatureCategory}s that relate to Hos.
   */
  public static final List<VehicleFeatureCategory> HOS_FEATURES = listOf(
      VehicleFeatureCategory.GEOTAB_DRIVE_HOS, VehicleFeatureCategory.GARMIN_HOS,
      VehicleFeatureCategory.HOS);

  /**
   * List of features currently active on the device.
   */
  private List<VehicleFeatureCategory> activeFeatures;

  /**
   * Gets a value indicating whether the device has access to the active tracking.
   */
  @JsonProperty("isActiveTrackingAllowed")
  private boolean isActiveTrackingAllowed;

  /**
   * Gets a value indicating if the device has access to engine functionality.
   */
  @JsonProperty("isEngineAllowed")
  private boolean isEngineAllowed;

  /**
   * Gets a value indicating whether the device has access to Garmin functionality.
   */
  @JsonProperty("isGarminAllowed")
  private boolean isGarminAllowed;

  /**
   * Gets a value indicating whether the device has access to HOS functionality.
   */
  @SuppressWarnings("AbbreviationAsWordInName")
  @JsonProperty("isHOSAllowed")
  private boolean isHOSAllowed;

  /**
   * Gets a value indicating whether the device has access to Iridium functionality.
   */
  @JsonProperty("isIridiumAllowed")
  private boolean isIridiumAllowed;

  /**
   * Gets a value indicating whether the device has access to odometer functionality.
   */
  @JsonProperty("isOdometerAllowed")
  private boolean isOdometerAllowed;

  /**
   * Gets a value indicating whether the device has access to trip detail functionality.
   */
  @JsonProperty("isTripDetailAllowed")
  private boolean isTripDetailAllowed;

  /**
   * Gets a value indicating whether the device can be viewed in MyGeotab.
   */
  @SuppressWarnings("AbbreviationAsWordInName")
  @JsonProperty("isUIAllowed")
  private boolean isUIAllowed;

  /**
   * Gets a value indicating whether the device has access to VIN functionality.
   */
  @SuppressWarnings("AbbreviationAsWordInName")
  @JsonProperty("isVINAllowed")
  private boolean isVINAllowed;

  /**
   * Gets the rate plans. Has been replaced by ActiveFeatures. Kept around for legacy serializers.
   */
  private List<String> ratePlans;

  public DeviceFlags(List<VehicleFeatureCategory> activeFeatures, List<VehicleFeatureCategory> allowedFeatures, List<String> restrictions) {
    for (String item : restrictions) {
      switch (item.toLowerCase(Locale.ROOT)) {
        case "nocheckmate":
          this.isUIAllowed = false;
          break;
        case "nocheckmatevin":
          this.isVINAllowed = false;
          break;
        case "nocheckmateodo":
          this.isOdometerAllowed = false;
          break;
        case "nocheckmateengine":
          this.isEngineAllowed = false;
          break;
        case "nocheckmatetripdetail":
          this.isTripDetailAllowed = false;
          break;
        default:
          break;
      }
    }
    this.isGarminAllowed = !Util.intersect(allowedFeatures, GARMIN_FEATURES).isEmpty();
    this.isHOSAllowed = !Util.intersect(allowedFeatures, HOS_FEATURES).isEmpty();
    this.isIridiumAllowed = allowedFeatures != null && allowedFeatures.contains(VehicleFeatureCategory.IRIDIUM);
    this.isActiveTrackingAllowed = allowedFeatures != null && allowedFeatures.contains(VehicleFeatureCategory.GO_ACTIVE);
    this.activeFeatures = Optional.ofNullable(activeFeatures).orElse(listOf());
  }
}
