package com.geotab.model.entity.device;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.geotab.util.DeviceDefaultsUtil;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * A GoCurve device. Additional properties can be seen in {@link GoDevice}
 */
@Data
@NoArgsConstructor
@SuperBuilder
public abstract class GoCurve extends GoDevice implements OdometerAdjustmentProviderAware,
    EngineHoursAdjustmentProviderAware {

  /**
   * The acceleration warning accelerometer threshold (y axis) value for the vehicle. A positive value that when
   * exceeded will trigger device beeping. Threshold value to mS2 conversion (threshold * 18 = milli-g / 1000 = g /
   * 1.0197162 = mS2). Default [22].
   */
  protected Integer accelerationWarningThreshold;

  /**
   * The accelerometer threshold warning factor value for this vehicle. Default [0].
   */
  protected Integer accelerometerThresholdWarningFactor;

  /**
   * The braking warning accelerometer threshold (y axis) value for the vehicle. A negative value that when exceeded
   * will trigger device beeping. Threshold value to mS2 conversion (threshold * 18 = milli-g / 1000 = g / 1.0197162 =
   * mS2). Default [-34].
   */
  protected Integer brakingWarningThreshold;

  /**
   * The cornering warning threshold (x axis) value for the vehicle. A positive value that when exceeded will trigger
   * device beeping (the additive inverse is automatically applied: 26/-26). Threshold value to mS2 conversion
   * (threshold * 18 = milli-g / 1000 = g / 1.0197162 = mS2). Default [26].
   */
  protected Integer corneringWarningThreshold;

  /**
   * Toggle to enable beeping when any of the acceleration thresholds are exceeded by device accelerometer readings.
   * Default [false].
   */
  @JsonProperty("enableBeepOnDangerousDriving")
  protected boolean enableBeepOnDangerousDriving;

  /**
   * Toggle to enable beeping when the vehicle's RPM exceeds the 'RpmValue'. Default [false].
   */
  @JsonProperty("enableBeepOnRpm")
  protected boolean enableBeepOnRpm;

  /**
   * The offset to be applied engine hours data reported by the engine computer. Default [0].
   */
  protected Integer engineHourOffset;

  /**
   * Toggle to enable active tracking on the device. enables Active Tracking which triggers the device to transmit data
   * more frequently. This allows for continuously up-to-date vehicle positions animated on the live map. It also
   * enables live server-side driver alerts.
   *
   * <p>Default [false].
   *
   * <p>This feature is only supported on the ProPlus plan. Turning on this feature on a device
   * using the Pro plan will automatically upgrade the device to the ProPlus plan with all associated charges.
   */
  @JsonProperty("isActiveTrackingEnabled")
  protected boolean isActiveTrackingEnabled;

  /**
   * Value which toggles beeping if an unbuckled seat belt is detected. This will only work if the device is able to
   * obtain seat belt information from the vehicle. Default [false].
   */
  @JsonProperty("isDriverSeatbeltWarningOn")
  protected boolean isDriverSeatbeltWarningOn;

  /**
   * Value which toggles monitoring both passenger and driver unbuckled seat belt, otherwise only the driver is
   * monitored. Default [false].
   */
  @JsonProperty("isPassengerSeatbeltWarningOn")
  protected boolean isPassengerSeatbeltWarningOn;

  /**
   * Value which toggles device beeping when the vehicle is reversing. Default [false].
   */
  @JsonProperty("isReverseDetectOn")
  protected boolean isReverseDetectOn;

  /**
   * Value which toggles device IOX USB connection. Default [true].
   */
  @JsonProperty("isIoxConnectionEnabled")
  protected boolean isIoxConnectionEnabled;

  /**
   * Used to correct the odometer value received from the engine. Default [1].
   */
  protected Float odometerFactor;

  /**
   * The offset to be applied odometer reported by the engine computer. Default [0].
   */
  protected Double odometerOffset;

  /**
   * The RPM value that when exceeded triggers device beeping. Default [3500].
   */
  protected Integer rpmValue;

  /**
   * The value in km/h that below will not trigger 'Seat belt Warning'. Default [10].
   */
  protected Double seatbeltWarningSpeed;

  @Override
  public void populateDefaults() {
    super.populateDefaults();
    DeviceDefaultsUtil.addGoCurveDefaults(this);
  }
}
