package com.geotab.model.entity.device;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.geotab.util.DeviceDefaultsUtil;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Device that supports curve logging and auxiliaries. Additional properties can be seen in {@link GoCurve}.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public abstract class GoCurveAuxiliary extends GoCurve implements DigitalAuxiliariesAware {

  /**
   * An array of the auxiliary warning speeds for the vehicle. The auxiliary is triggered when the speed is greater than
   * or equal to this value. Maximum length [8] Default [0,0,0,0,0,0,0,0].
   */
  protected double[] auxWarningSpeed;

  /**
   * Toggle to enable auxiliary warnings.
   *
   * <p>Maximum length [8] Default [false,false,false,false,false,false,false,false].
   */
  protected boolean[] enableAuxWarning;

  /**
   * Toggle to enable control external relay value for the vehicle. Default [false].
   */
  protected boolean enableControlExternalRelay;

  /**
   * The option which controls how long any attached external devices (Garmin, Iridium, HOS, RFID, RS232, CAN, and USB)
   * are kept on after the vehicle is turned off in minutes. Default [0].
   */
  protected int externalDeviceShutDownDelay;

  /**
   * It is used to define the delay before the driver identification reminder is sent out if the driver key has not been
   * not swiped. The maximum value of this property is 255. When it is less or equal to 180, it indicates the number of
   * seconds of the delay.  When it is greater than 180, the delay increases 30 seconds for every increment of one of
   * this property. For example, 180 indicates 180 seconds, 181 indicates 210 seconds, and 182 indicates 240 seconds.
   * Maximum [255] Default [30].
   */
  protected int immobilizeArming;

  /**
   * Value mainly used for enable or disable driver identification reminder. If it is used in conjunction with vehicle
   * relay circuits, it can force the driver to swipe the driver key before starting the vehicle. Default [false].
   */
  protected boolean immobilizeUnit;

  /**
   * An array of the IsAuxIgnTriggers for the vehicle.
   */
  @JsonProperty("isAuxIgnTrigger")
  protected boolean[] isAuxIgnTrigger;

  /**
   * An array of boolean flags indicating if a corresponding Aux signal should be inverted on importing the device
   * data.
   *
   * <p>Maximum length [8] Default [false,false,false,false,false,false,false,false].
   */
  @JsonProperty("isAuxInverted")
  protected boolean[] isAuxInverted;

  public GoCurveAuxiliary setAuxWarningSpeed(double[] auxWarningSpeed) {
    this.auxWarningSpeed = checkAndPadAuxArray(auxWarningSpeed);
    return this;
  }

  public GoCurveAuxiliary setEnableAuxWarning(boolean[] enableAuxWarning) {
    this.enableAuxWarning = checkAndPadAuxArray(enableAuxWarning);
    return this;
  }

  public GoCurveAuxiliary setAuxIgnTrigger(boolean[] isAuxIgnTrigger) {
    this.isAuxIgnTrigger = checkAndPadAuxIgnArray(isAuxIgnTrigger);
    return this;
  }

  public GoCurveAuxiliary setIsAuxInverted(boolean[] isAuxInverted) {
    this.isAuxInverted = checkAndPadAuxArray(isAuxInverted);
    return this;
  }

  @Override
  public void populateDefaults() {
    super.populateDefaults();
    DeviceDefaultsUtil.addAuxiliaryDefaults(this);
  }
}
